/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <kstat.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/swap.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>

#include "libsysactivity.h"
#include "utils.h"

static int populate_swaptable(uint16_t nswt_ent);
static void free_swaptable_entries();

__thread kstat_ctl_t* memory_kc;
__thread int pagesize = 0;
__thread swaptbl_t* swaptable;
__thread size_t swaptable_size;

int sa_open_memory() {
	swaptable = NULL;
	swaptable_size = 0;

	memory_kc = kstat_open();
	if (memory_kc == NULL)
		return ENOSYS;

	pagesize = sysconf(_SC_PAGESIZE);
	return 0;
}

int sa_close_memory() {
	if (memory_kc != NULL)
		kstat_close(memory_kc);
	if (swaptable != NULL)
		free(swaptable);

	return 0;
}

int sa_get_memory(struct sa_memory* dst) {
	if (dst == NULL)
		return EINVAL;

	kstat_t* ks;
	if (get_ks(memory_kc, &ks, "unix", "system_pages") != 0)
		return ENOSYS;

	kstat_named_t* kn = kstat_data_lookup(ks, "pagestotal");
	if (kn == NULL)
		return ENOSYS;
	dst->total = (uint64_t) kn->value.ui64 * pagesize;

	kn = kstat_data_lookup(ks, "pagesfree");
	if (kn == NULL)
		return ENOSYS;
	dst->free = (uint64_t) kn->value.ui64 * pagesize;

	kn = kstat_data_lookup(ks, "pageslocked");
	if (kn == NULL)
		return ENOSYS;
	dst->locked = (uint64_t) kn->value.ui64 * pagesize;

	struct anoninfo swap_info; 
	if (swapctl(SC_AINFO, &swap_info) == -1)
		return ENOSYS; 
	dst->swap_total = swap_info.ani_max;
	dst->swap_free = swap_info.ani_free; 

	return 0;
}

int sa_count_swaps(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	int ret = swapctl(SC_GETNSWP, NULL);
	if (ret == -1)
		return errno;

	*number = (uint16_t) ret;
	return 0;
}

int sa_get_swap(uint16_t index, struct sa_swap* dst) {
	if (dst == NULL)
		return EINVAL;

	int nswaps = swapctl(SC_GETNSWP, NULL);
	if (nswaps == -1)
		return errno;

	if (index >= nswaps)
		return ENODEV;

	int ret = populate_swaptable(index + 1);
	if (ret != 0)
		return ret;

	strncpy(dst->name, swaptable->swt_ent[index].ste_path, sizeof(dst->name));
	dst->total = swaptable->swt_ent[index].ste_pages * pagesize;
	dst->free = swaptable->swt_ent[index].ste_free * pagesize;

	free_swaptable_entries();
	return 0;
}

int sa_get_swaps(struct sa_swap* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	int nswaps = swapctl(SC_GETNSWP, NULL);
	if (nswaps == -1)
		return errno;

	int ret = populate_swaptable(nswaps);
	if (ret != 0)
		return ret;

	uint16_t i;
	for (i = 0; i < dst_size; i++) {
		if (i == nswaps) {
			free_swaptable_entries();
			return ENODEV;
		}

		strncpy(dst[i].name, swaptable->swt_ent[i].ste_path, sizeof(dst->name));
		dst[i].total = swaptable->swt_ent[i].ste_pages * pagesize;
		dst[i].free = swaptable->swt_ent[i].ste_free * pagesize;
		(*written)++;
	}

	free_swaptable_entries();
	return 0;
}

static int populate_swaptable(uint16_t nswt_ent) {
	int len = sizeof(int) + (nswt_ent * sizeof(struct swapent));
	if (len > swaptable_size) {
		swaptable = (swaptbl_t*) realloc(swaptable, len);
		if (swaptable == NULL)
			return ENOMEM;
		swaptable_size = len;
	}

	swaptable->swt_n = nswt_ent;

	int i;
	for (i = 0; i < nswt_ent; i++) {
		swaptable->swt_ent[i].ste_path = (char*) malloc(MAXPATHLEN);
		if (swaptable->swt_ent[i].ste_path == NULL)
			return ENOMEM;
	}

	if (swapctl(SC_LIST, swaptable) == -1)
		return errno;
	return 0;
}

static void free_swaptable_entries() {
	int i;
	for (i = 0; i < swaptable->swt_n; i++) {
		free(swaptable->swt_ent[i].ste_path);
	}
}
