/** -*- C++ -*-
    @file cache/relation.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <apt-pkg/version.h>

#include <wibble/shared.h>
#include <ept/forward.h>
#include <ept/cache/entity.h>
#include <stdexcept>
#include <ept/cache/apt/index.h>

#ifndef EPT_CACHE_RELATION
#define EPT_CACHE_RELATION

namespace ept {
namespace t {
namespace cache {

template< typename C >
struct Relation : public EntityMixin< Relation< C > >
{
    enum Type { Dependency = 0, PreDependency = 1, Suggests = 2, Recommends = 3,
                Conflicts = 5, Replaces = 6, Obsoletes = 7 };

    typedef typename C::RelationPointer RelationPointer;
    typedef typename C::Package Package;
    typedef typename C::Version Version;

    const typename C::Aggregator &aggregator() const { return m_pointer.aggregator(); }

    Relation() {}
    Relation( RelationPointer p ) : m_pointer( p ) {}

    Relation nextInCache() const {
        Relation r = *this;
        r.m_pointer.m_dependency = lastAtomInGroup(); // XXX
        return r;
    }

    long id() const { return reinterpret_cast< long >( m_pointer.m_dependency ); } // XXX for
                                                       // comparison
                                                       // only(!)

    /* Interface safety. */
    bool valid() const {
        return m_pointer.valid();
        /* return m_cache && dataValid( m_data )
           && type() >= 0 && type() <= 7; */
    }

    /**
       @brief Get a sensible name of the relation.

       This will give you a reasonably useful name of the relation's declaration.
       It does NOT include the type. For a given package/version/type combination,
       this should also be fairly unique. It roughly corresponds how apt-cache
       formats the Depends: et al fields when showing package details.
    */
    std::string name() const;
    std::string format() const;
    std::string typeString() const;

    /**
       @brief Get list of possible targets for this relation.

       This method will return you a list of all VersionIterator's that are
       fit for satisfying this relation declaration. For depends, this means
       list of packages that would satisfy the dependency (you need to have
       only once of those installed to satisfy it!), for conflicts, this means
       list of all packages conflicting with owner of this relation (you
       need to uninstall all of those to be able to install owner!).
    */
    wibble::Range< Version > targetList() const;
    // the above should be implemented using targetPackages()

    wibble::Range< Package > targetPackages() const {
        wibble::SharedVector< Package > &r = wibble::shared< wibble::SharedVector< Package > >();
        Atom a = firstAtom();
        while ( a.valid() ) {
            Provides p( *this, a.package() );
            // XXX libapt-pkg foo
            if ( a.package().candidateVersion().valid()
                 && aggregator().state().aptState().VS().CheckDep(
                    a.package().candidateVersion().versionString().c_str(),
                    a.compareOperator(), a.targetVersionString().c_str() ) ) {
                wibble::Range< Version > vr = a.package().versions();
                consumer( r ).consume( a.package() );
            }
            while ( !p.isLast() ) {
                /* std::cerr << "looking at providing pkg: "
                   << p.package().name() << std::endl; */
                if ( p.package().hasVersion() )
                    consumer( r ).consume( p.package() );
                p.advance();
            }
            a = a.next();
        }
        return wibble::uniqueRange( wibble::range( r ) );
    }

    /// Get the type of this relation.
    Type type() const;

    /// Get the owner PackageIterator (that is, the one from which we were obtained).
    Package ownerPackage() const;

    /// Get the owner VersionIterator (that is the one from which we were obtained).
    Version ownerVersion() const {
        return aggregator().index().versionOwningDependency( firstAtom().d );
    }

    /**
       @brief Get a "straight" version of this RelationIterator.

       This will return a straight (as opposed to reverse) version of this
       very same iterator. If it was already straight, you will get a copy
       of it. If it was reverse, the straight variant will be looked up in the
       cache and returned.
    */
    Relation straight() const;

    bool isReverse() const { // XXX
        return false;
    }

protected:
    pkgCache::Dependency *dependency() const { return m_pointer.dependency(); }
    RelationPointer m_pointer;

    struct Atom {

        Package package() const {
            if ( !valid() )
                throw std::out_of_range( "past-the-end Atom dereferenced" );
            return r.aggregator().index().packageFromDependency( d );
        }

        bool valid() const {
            return r.dataValid( d );
        }

        Atom next() const {
            if ( !valid() )
                throw std::out_of_range( "past-the-end Atom dereferenced" );
            if ( r.nextAtomBelongsToGroup( d ) )
                return Atom( r, r.nextAtomInCache( d ) );
            return Atom( r, 0 );
        }

        std::string name() const { return package().name(); }
        std::string format() const;

        int compareOperator() const {
            return d->CompareOp;
        }

        std::string targetVersionString() const {
            return std::string( r.aggregator().index().strPtr( d->Version ) );
        }

        Atom( const Relation &_r, pkgCache::Dependency *_d )
            : d( _d ), r( _r )
        {}

        Atom &operator=( const Atom &a ) {
            if ( a.r != r )
                throw std::bad_cast(); // XXX
            d = a.d;
            return *this;
        }

        pkgCache::Dependency *d;
        const Relation &r;
    };

    Atom firstAtom() const {
        return Atom( *this, dependency() );
    }

    pkgCache::Dependency *nextAtomInCache( pkgCache::Dependency *c ) const {
        pkgCache::Dependency *r = c;
        map_ptrloc off = 0;
        if (isReverse())
            off = r->NextRevDepends;
        else
            off = r->NextDepends;
        r = aggregator().index().depPtr( off );
        return r;
        /* if (r->Type == c->Type)
           break;
        if ( !dataValid( r ) )
            return r;
        n&& r->Type == c->Type )
            return r;
        if ( d == ownerVersion().firstDependInCache().firstAtom().d )
        break; */
    }

    pkgCache::Dependency *lastAtomInGroup() const {
        pkgCache::Dependency *r = dependency();
        bool run = true;
        while (run) {
            run = nextAtomBelongsToGroup( r );
            r = nextAtomInCache( r );
            if (not dataValid( r ) )
                run = false;
        }
        // std::cerr << "data valid: " << dataValid( r ) << std::endl;
        return r;
    }

    bool nextAtomBelongsToGroup( pkgCache::Dependency *d ) const {
        return ((d->CompareOp & pkgCache::Dep::Or) == pkgCache::Dep::Or);
    }

    bool dataValid( pkgCache::Dependency *d ) const {
        return d && d != aggregator().index().depPtr();
        // && d != ownerVersion().firstDependInCache().firstAtom().d ;
    }

public:
    struct Provides { // XXX
        enum Type { TVersion, TPackage };

        Provides( const Relation &_r, Version v )
            : p( _r.aggregator().index().providesPtr( v.data()->ProvidesList ) ),
              r( _r ), t( TVersion )
        {}

        Provides( const Relation &_r, Package _p )
            : p( _r.aggregator().index().providesPtr(
                     _p.pointer().package()->ProvidesList ) ),
              r( _r ), t( TPackage )
        {}

        Provides next() {
            Provides x = *this;
            x.advance();
            return x;
        }

        // XXX argh
        Package package() { return r.aggregator().index().versionFromProvides( p ).package(); }

        void advance() {
            p = r.aggregator().index().providesPtr(
                t == TVersion ? p->NextPkgProv : p->NextProvides );
        }

        bool isLast() { return p == r.aggregator().index().providesPtr(); }
        pkgCache::Provides *p;
        const Relation &r;
        Type t;
    };

};

}
}
}

#endif
