#include <ept/cache/debtags/update.h>
#include <ept/cache/debtags/vocabularymerger.h>
#include <ept/path.h>

#include <wibble/exception.h>
#include <wibble/sys/fs.h>

#include <tagcoll/input/zlib.h>
#include <tagcoll/input/stdio.h>

#include <sys/stat.h>

namespace ept {
namespace t {
namespace cache {
namespace debtags {

Directory::const_iterator Directory::begin()
{
	DIR* dir = opendir(m_path.c_str());
	if (!dir)
		throw wibble::exception::System("reading directory " + m_path);
	return const_iterator(dir);
}

Directory::const_iterator Directory::end() const
{
	return const_iterator();
}

bool Directory::valid()
{
	// Check that the directory exists
	std::auto_ptr<struct stat> st = wibble::sys::fs::stat(path());
	if (st.get() == NULL)
		return false;
	// Check that it is a directory
	if (!S_ISDIR(st->st_mode))
		return false;
	return true;
}


SourceDir::FileType SourceDir::fileType(const std::string& name)
{
	if (name[0] == '.') return SKIP;

	// Filenames need to be at least 5 characters long (one char plus
	// extension)
	if (name.size() <= 4) return SKIP;

	// Only look at .voc and .tag files
	std::string ext(name, name.size() - 4);
	if (ext == ".voc")
		return VOC;
	if (ext == ".tag")
		return TAG;

	// Now look for compressed files, which must have the 4 character extension
	// plus the 3 chars of '.gz'
	if (name.size() <= 7) return SKIP;

	ext = name.substr(name.size() - 7);
	if (ext == ".voc.gz")
		return VOCGZ;
	if (ext == ".tag.gz")
		return TAGGZ;

	return SKIP;
}

time_t SourceDir::timestamp()
{
	if (!valid()) return 0;

	time_t max = 0;
	for (const_iterator d = begin(); d != end(); ++d)
	{
		FileType type = fileType(d->d_name);
		if (type == SKIP) continue;

		time_t ts = Path::timestamp(path() + "/" + d->d_name);
		if (ts > max) max = ts;
	}

	return max;
}

time_t SourceDir::vocTimestamp()
{
	if (!valid()) return 0;

	time_t max = 0;
	for (const_iterator d = begin(); d != end(); ++d)
	{
		FileType type = fileType(d->d_name);
		if (type != VOC and type != VOCGZ) continue;

		time_t ts = Path::timestamp(path() + "/" + d->d_name);
		if (ts > max) max = ts;
	}

	return max;
}

time_t SourceDir::tagTimestamp()
{
	if (!valid()) return 0;

	time_t max = 0;
	for (const_iterator d = begin(); d != end(); ++d)
	{
		FileType type = fileType(d->d_name);
		if (type != TAG and type != TAGGZ) continue;

		time_t ts = Path::timestamp(path() + "/" + d->d_name);
		if (ts > max) max = ts;
	}

	return max;
}

void SourceDir::readVocabularies(VocabularyMerger& out)
{
	if (!valid()) return;

	for (const_iterator d = begin(); d != end(); ++d)
	{
		FileType type = fileType(d->d_name);
		if (type == VOC)
		{
			// Read uncompressed data
			tagcoll::input::Stdio in(path() + "/" + d->d_name);

			// Read the vocabulary
			out.read(in);
		}
		else if (type == VOCGZ)
		{
			// Read compressed data
			tagcoll::input::Zlib in(path() + "/" + d->d_name);

			// Read the vocabulary
			out.read(in);
		}
	}
}

#if 0
SourceDir SourceDir::main()
{
	return SourceDir(Path::stateDir());
}

SourceDir SourceDir::user()
{
	return SourceDir(Path::rcdir(false));
}
#endif


}
}
}
}

#include <ept/cache/debtags/update.tcc>

// vim:set ts=4 sw=4:
