/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* file:     parametric_2d.c                                                */
/*                                                                          */
/*                                                                          */
/* description: Support for parametric elements in 2D                       */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Daniel Koester                                               */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by D. Koester (2005)                                                */
/*--------------------------------------------------------------------------*/

static const REAL_B mid_lambda_2d = { 0.5, 0.5, 0.0 };
static const REAL_B bary2_2d[6] = {{1.0, 0.0, 0.0, 0.0},
				   {0.0, 1.0, 0.0, 0.0},
				   {0.0, 0.0, 1.0, 0.0},
				   {0.0, 0.5, 0.5, 0.0},
				   {0.5, 0.0, 0.5, 0.0},
				   {0.5, 0.5, 0.0, 0.0}};


/*--------------------------------------------------------------------------*/
/* Functions for affine elements as parametric elements. (suffix 1_2d)      */
/*--------------------------------------------------------------------------*/

static void det1_2d(const EL_INFO *el_info, const QUAD *quad, int N,
		    const REAL lambda[][N_LAMBDA], REAL dets[])
{
  REAL_D *local_coords = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data)->local_coords;
  REAL        e1[DIM_OF_WORLD], e2[DIM_OF_WORLD], det;
  int         i, n;

  for (i = 0; i < DIM_OF_WORLD; i++)
  {
    e1[i] = local_coords[1][i] - local_coords[0][i];
    e2[i] = local_coords[2][i] - local_coords[0][i];
  }

#if DIM_OF_WORLD==2
  det = WEDGE_DOW(e1, e2);
  det = ABS(det);
#else
  {
    REAL_D normal;

    WEDGE_DOW(e1, e2, normal);
    det = NORM_DOW(normal);
  }
#endif
  if (quad) N = quad->n_points;

  for (n = 0; n < N; n++)
    dets[n] = det;

  return;
}


static void grd_lambda1_2d(const EL_INFO *el_info, const QUAD *quad,
			int N, const REAL lambda[][N_LAMBDA],
			REAL_D grd_lam[][N_LAMBDA], REAL dets[])
{
  FUNCNAME("grd_lambda1_2d");
  REAL_D *local_coords = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data)->local_coords;
  int         i, j, n;
  REAL        e1[DIM_OF_WORLD], e2[DIM_OF_WORLD], adet;
#if DIM_OF_WORLD == 2
  REAL        det;
  REAL        a11, a12, a21, a22;
#endif
  REAL_D      grd_lambda[N_LAMBDA];

  for (i = 0; i < DIM_OF_WORLD; i++)
  {
    e1[i] = local_coords[1][i] - local_coords[0][i];
    e2[i] = local_coords[2][i] - local_coords[0][i];
  }

#if DIM_OF_WORLD == 2
  det = WEDGE_DOW(e1, e2);
  adet = ABS(det);
  if (adet < 1.0E-15) 
  {
    MSG("abs(det) = %lf\n", adet);
    for (i = 0; i <= 2; i++)
      for (j = 0; j < DIM_OF_WORLD; j++)
	grd_lambda[i][j] = 0.0;
  }
  else
  {
    det = 1.0 / det;
    a11 =  e2[1] * det;       /* (a_ij) = A^{-T} */
    a21 = -e2[0] * det;
    a12 = -e1[1] * det;
    a22 =  e1[0] * det;

    grd_lambda[1][0] = a11;
    grd_lambda[1][1] = a21;
    grd_lambda[2][0] = a12;
    grd_lambda[2][1] = a22;
    grd_lambda[0][0] = - grd_lambda[1][0] - grd_lambda[2][0];
    grd_lambda[0][1] = - grd_lambda[1][1] - grd_lambda[2][1];
  }
#endif

#if DIM_OF_WORLD == 3
  {
    REAL_D normal;
    
    WEDGE_DOW(e1, e2, normal);
    adet = SCP_DOW(normal,normal);

    if (adet < 1.0E-15) 
    {
      MSG("abs(det) = %lf\n", adet);
      for (i = 0; i <= 2; i++)
	for (j = 0; j < DIM_OF_WORLD; j++)
	  grd_lambda[i][j] = 0.0;
    }
    else
    {
      WEDGE_DOW(e2, normal, grd_lambda[1]);
      WEDGE_DOW(normal, e1, grd_lambda[2]);
      
      for (i = 0; i < DIM_OF_WORLD; i++)
      {
	grd_lambda[1][i] /= adet;
	grd_lambda[2][i] /= adet;
      }
      grd_lambda[0][0] = - grd_lambda[1][0] - grd_lambda[2][0];
      grd_lambda[0][1] = - grd_lambda[1][1] - grd_lambda[2][1];
      grd_lambda[0][2] = - grd_lambda[1][2] - grd_lambda[2][2];
      
      adet = sqrt(adet);
    }
  }
#endif

#if 0
  MSG("el %p : det = %8.6lf ----------------\n",
      el_info->el, adet);
  for (i = 0; i < N_VERTICES_2D; i++)
    MSG("vertex     %1d = (%8.4lf, %8.4lf)\n",i,
	local_coords[i][0], local_coords[i][1]);
#if DIM_OF_WORLD == 2
  for (i = 0; i < N_VERTICES_2D; i++)
    MSG("grd_lam %1d = (%8.4lf, %8.4lf)\n",i,
	grd_lambda[i][0], grd_lambda[i][1]);
#else
  for (i = 0; i < N_VERTICES_2D; i++)
    MSG("grd_lam %1d = (%8.4lf, %8.4lf, %8.4lf)\n",i,
	grd_lambda[i][0], grd_lambda[i][1], grd_lambda[i][2]);
#endif
#endif

  if (quad) N = quad->n_points;
  for (n = 0; n < N; n++)
  {
    for (i = 0; i <= 2; i++)
      for (j = 0; j < DIM_OF_WORLD; j++)
	grd_lam[n][i][j] = grd_lambda[i][j];
    dets[n] = adet;
  }

  return;
}


/****************************************************************************/
/* fill_coords1_2d(data): initialize the DOF_REAL_D_VEC coords containing   */
/* the position data of the parametric elements (coordinates of vertices in */
/* this case).                                                              */
/****************************************************************************/

static void fill_coords1_2d(LAGRANGE_PARAM_DATA *data)
{
  DOF_REAL_D_VEC  *coords = data->coords;
  DOF_UCHAR_VEC   *touched_coords = data->touched_coords;
  NODE_PROJECTION *n_proj = data->n_proj, *active_proj;
  TRAVERSE_STACK  *stack = get_traverse_stack();
  const EL_INFO   *el_info;
  FLAGS            fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_PROJECTION;
  const DOF     *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;
  int              i, n;

  get_dof = coords->fe_space->bas_fcts->get_dof_indices;
  admin   = coords->fe_space->admin;

  el_info = traverse_first(stack, coords->fe_space->mesh, -1, fill_flag);
  while (el_info) {
    const DOF *dof = (*get_dof)(el_info->el, admin, nil);
    REAL      *vec;

    for (i = 0; i < N_VERTICES_2D; i++) {
      vec = coords->vec[dof[i]];
      for (n = 0; n < DIM_OF_WORLD; n++)
	vec[n] = el_info->coord[i][n];

      /* Look for a projection function that applies to vertex[i]. */
      /* Apply this projection if found.                           */

      if(touched_coords)
	touched_coords->vec[dof[i]] = 0;

      if(n_proj && (n_proj->func)) {
	active_proj = el_info->projections[1 + (i+1)%3];
	if(!active_proj) active_proj = el_info->projections[1 + (i+2)%3];

	if(!active_proj) active_proj = el_info->projections[0];

	if(active_proj == n_proj) {
	  n_proj->func(vec, el_info, bary2_2d[i]);

	  if(touched_coords)
	    touched_coords->vec[dof[i]] = 1;
	}
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);
  
  return;
}


/****************************************************************************/
/* refine_interpol1_2d(drdv,list,n): update coords vector during refinement.*/
/****************************************************************************/

static void refine_interpol1_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list, int n)
{
  FUNCNAME("refine_interpol1_2d");
  EL                      *el;
  REAL_D                  *vec = nil;
  DOF                      dof_new, dof0, dof1;
  LAGRANGE_PARAM_DATA     *data = 
    (LAGRANGE_PARAM_DATA *)drdv->fe_space->mesh->parametric->data;
  NODE_PROJECTION         *n_proj = data->n_proj;
  DOF_UCHAR_VEC           *touched_coords = data->touched_coords;
  int                      n0, j;

  GET_DOF_VEC(vec, drdv);
  n0 = drdv->fe_space->admin->n0_dof[VERTEX];
  el = list->el_info.el;

  dof0 = el->dof[0][n0];           /* 1st endpoint of refinement edge */
  dof1 = el->dof[1][n0];           /* 2nd endpoint of refinement edge */
  dof_new = el->child[0]->dof[2][n0];   /* vertex[2] is newest vertex */

  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[dof_new][j] = 0.5*(vec[dof0][j] + vec[dof1][j]);

  if(touched_coords)
    touched_coords->vec[dof_new] = 0;

  if(n_proj && n_proj->func) {
    if(list->el_info.active_projection == n_proj) {
      n_proj->func(vec[dof_new], &list->el_info, mid_lambda_2d);
      
      if(touched_coords)
	touched_coords->vec[dof_new] = 1;
    }
  }
    
  return;
}


/*--------------------------------------------------------------------------*/
/* Common functions for higher order elements. (suffix y_2d)                */
/*--------------------------------------------------------------------------*/

#define N_BAS2_2D  6

typedef struct DD_data_2d DD_DATA_2D;
struct DD_data_2d
{
  const QUAD *quad;

  int         n_bas_fcts;
  REAL      (*DD)[N_BAS2_2D][2];

  DD_DATA_2D *next;
};

static DD_DATA_2D *init_dd_data_2d(const QUAD *quad, const BAS_FCTS *bas_fcts)
{
  FUNCNAME("init_dd_data_2d");
  static DD_DATA_2D *first_dd_data = nil;
  DD_DATA_2D        *data;
  int                iq, i;
  const REAL        *grd;

  DEBUG_TEST_EXIT(bas_fcts->n_bas_fcts <= N_BAS2_2D,
    "Sorry, only up to 6 local DOFs at the moment.\n");

  for (data = first_dd_data; data; data = data->next)
    if (data->quad == quad) break;

  if (data)  return(data);

  data = MEM_ALLOC(1, DD_DATA_2D);

  data->quad       = quad;
  data->n_bas_fcts = bas_fcts->n_bas_fcts;
  data->next       = first_dd_data;

  first_dd_data = data;

  data->DD = (REAL (*)[N_BAS2_2D][2])
    alberta_alloc(quad->n_points*sizeof(REAL [N_BAS2_2D][2]),
		  funcName, __FILE__,  __LINE__);

  for (iq = 0; iq < quad->n_points; iq++) {
    for (i = 0; i < bas_fcts->n_bas_fcts; i++) {
      grd = (*bas_fcts->grd_phi[i])(quad->lambda[iq]);

      data->DD[iq][i][0] = -grd[0] + grd[1];
      data->DD[iq][i][1] = -grd[0] + grd[2];
    }
  }

  return(data);
}

/*--------------------------------------------------------------------------*/
/*---  compute D^t and D^t D, return det(D^t D)                          ---*/
/*--------------------------------------------------------------------------*/

static REAL Dt_and_DtD_2d(REAL_D *F, REAL DD[][2], int n_bas,
			  REAL Dt[2][DIM_OF_WORLD], REAL DtD[2][2])
{
  FUNCNAME("Dt_and_DtD_2d");
  int   i, j, n, m;
  REAL  val;

  for (m = 0; m < 2; m++) {
    for (n = 0; n < DIM_OF_WORLD; n++) {
      for (Dt[m][n] = i = 0; i < n_bas; i++)
	Dt[m][n] += F[i][n]*DD[i][m];
    }
  }

  for (i = 0; i < 2; i++) {
    for (j = i; j < 2; j++) {
      for (val = n = 0; n < DIM_OF_WORLD; n++)
	val += Dt[i][n]*Dt[j][n];
      DtD[i][j] = DtD[j][i] = val;
    }
  }

  val = DtD[0][0]*DtD[1][1] - DtD[0][1]*DtD[1][0];

  if (val < 0.0) {
    WARNING("val = %e\n", val);
    for (i = 0; i < n_bas; i++)
      PRINT_REAL_VEC("F", F[i], DIM_OF_WORLD);
    WAIT_REALLY;
  }

  return(val);
}

static void dety_2d(const EL_INFO *el_info, const QUAD *quad, int N,
		    const REAL lambda[][N_LAMBDA], REAL dets[])
{
  /*FUNCNAME("dety_2d");*/
  REAL                      Dt[2][DIM_OF_WORLD], DtD[2][2];
  int                       iq;
  const BAS_FCTS           *bas_fcts;
  LAGRANGE_PARAM_DATA      *data = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data);
  REAL_D                   *local_coords = data->local_coords;

  /* First of all, check if we are on a parametric simplex.            */
  /* If not, treat this simplex as an affine simplex, even though some */
  /* higher order vertices might be shifted.                           */
  if(data->i_am_affine) {
    det1_2d(el_info, quad, N, lambda, dets);
    return;
  }

  if (quad) {
    static DD_DATA_2D *dd_data = nil;

    if (!dd_data  ||  dd_data->quad != quad) {
      bas_fcts = data->coords->fe_space->bas_fcts;

      dd_data = init_dd_data_2d(quad, bas_fcts);
    }

    for (iq = 0; iq < quad->n_points; iq++)
      dets[iq] = 
	sqrt(Dt_and_DtD_2d(local_coords, dd_data->DD[iq],
			   dd_data->n_bas_fcts, Dt, DtD));
  }
  else {
    REAL  DD[N_BAS2_2D][2];
    const REAL *grd;
    int   i, n_bas;

    bas_fcts = data->coords->fe_space->bas_fcts;
    n_bas = bas_fcts->n_bas_fcts;

    for (iq = 0; iq < N; iq++) {
      for (i = 0; i < n_bas; i++) {
	grd = (*bas_fcts->grd_phi[i])(lambda[iq]);

	DD[i][0] = -grd[0] + grd[1];
	DD[i][1] = -grd[0] + grd[2];
      }
      dets[iq] = sqrt(Dt_and_DtD_2d(local_coords, DD, n_bas, Dt, DtD));
    }
  }

  return;
}

static REAL Lambda_iq_2d(REAL_D *const F, REAL DD[][2], int n_bas,
			 REAL_D Lambda[N_LAMBDA])
{
  REAL    Dt[2][DIM_OF_WORLD], DtD[2][2];
  REAL    DtD_1[2][2], DFS_1[2][DIM_OF_WORLD];
  REAL    det, det_1;
  int     n;

  det = Dt_and_DtD_2d(F, DD, n_bas, Dt, DtD);
  det_1 = 1.0/det;

/*--------------------------------------------------------------------------*/
/*--- now, invert DtD                                                    ---*/
/*--------------------------------------------------------------------------*/
  DtD_1[0][0] = det_1*DtD[1][1];
  DtD_1[1][1] = det_1*DtD[0][0];
  DtD_1[0][1] = DtD_1[1][0] = -det_1*DtD[0][1];

/*--------------------------------------------------------------------------*/
/*--- compute DF_S^{-1} = D^t D^{-1} D^t                                 ---*/
/*--------------------------------------------------------------------------*/
  
  for (n = 0; n < DIM_OF_WORLD; n++) {
    DFS_1[0][n] = DtD_1[0][0]*Dt[0][n] + DtD_1[0][1]*Dt[1][n];
    DFS_1[1][n] = DtD_1[1][0]*Dt[0][n] + DtD_1[1][1]*Dt[1][n];
  }

/*--------------------------------------------------------------------------*/
/*--- finally, \Lambda = \hat\Lambda DF_S^{-1}                           ---*/
/*--------------------------------------------------------------------------*/

  for (n = 0; n < DIM_OF_WORLD; n++) {
    Lambda[0][n] = -DFS_1[0][n] - DFS_1[1][n];
    Lambda[1][n] = DFS_1[0][n];
    Lambda[2][n] = DFS_1[1][n];
  }

  return(sqrt(det));
}

static void grd_lambday_2d(const EL_INFO *el_info, const QUAD *quad,
			   int N, const REAL lambda[][N_LAMBDA],
			   REAL_D Lambda[][N_LAMBDA], REAL dets[])
{
#if 0
  FUNCNAME("grd_lambday_2d");
#endif
  int                       iq;
  const BAS_FCTS           *bas_fcts;
  LAGRANGE_PARAM_DATA      *data = 
    ((LAGRANGE_PARAM_DATA *)el_info->mesh->parametric->data);
  REAL_D                   *local_coords = data->local_coords;

  /* First of all, check if we are on a parametric simplex.            */
  /* If not, treat this simplex as an affine simplex, even though some */
  /* higher order vertices might be shifted.                           */
  if(data->i_am_affine) {
    grd_lambda1_2d(el_info,quad, N, lambda, Lambda, dets);
    return;
  }

  if (quad) {
    static DD_DATA_2D *dd_data = nil;

    if (!dd_data  ||  dd_data->quad != quad) {
      bas_fcts = data->coords->fe_space->bas_fcts;

      dd_data = init_dd_data_2d(quad, bas_fcts);
    }

    for (iq = 0; iq < quad->n_points; iq++)
      dets[iq] = Lambda_iq_2d(local_coords, dd_data->DD[iq],
			   dd_data->n_bas_fcts, Lambda[iq]);
  }
  else
  {
    REAL  DD[N_BAS2_2D][2];
    const REAL *grd;
    int   i, n_bas;

    bas_fcts = data->coords->fe_space->bas_fcts;
    n_bas = bas_fcts->n_bas_fcts;

    for (iq = 0; iq < N; iq++) {
      for (i = 0; i < n_bas; i++) {
	grd = (*bas_fcts->grd_phi[i])(lambda[iq]);

	DD[i][0] = -grd[0] + grd[1];
	DD[i][1] = -grd[0] + grd[2];
      }
      dets[iq] = Lambda_iq_2d(local_coords, DD, n_bas, Lambda[iq]);
    }
  }

#if 0
  int n;
  
  MSG("el %p : ----------------\n",
      el_info->el);
  for (n = 0; n < N_VERTICES_2D+N_EDGES_2D; n++)
    MSG("vertex     %1d = (%8.4lf, %8.4lf)\n",n,
	local_coords[n][0], local_coords[n][1]);
#if DIM_OF_WORLD == 2
  for (n = 0; n < N_VERTICES_2D+N_EDGES_2D; n++)
    MSG("grd_lam %1d = (%8.4lf, %8.4lf)\n",n,
	Lambda[n][0], Lambda[n][1]);
#else
  for (n = 0; n < N_VERTICES_2D+N_EDGES_2D; n++)
    MSG("grd_lam %1d = (%8.4lf, %8.4lf, %8.4lf)\n",n,
	Lambda[n][0], Lambda[n][1], Lambda[n][2]);
#endif
#endif

  return;
}


/*--------------------------------------------------------------------------*/
/* Functions for quadratic elements. (suffix 2_2d)                          */
/*--------------------------------------------------------------------------*/

static void refine_interpol2_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list, int n)
{
  FUNCNAME("refine_interpol2_2d");
  EL                  *el;
  REAL_D              *vec = nil;
  int                  node_v, node_e, n0_v, n0_e, i, j, result;
  DOF                  cdof[3], cdof_e[N_EDGES_2D], cdof_v[N_VERTICES_2D],pdof;
  REAL_D               x[3];
  NODE_PROJECTION     *n_proj;
  const MESH          *mesh = drdv->fe_space->mesh;
  LAGRANGE_PARAM_DATA *data = (LAGRANGE_PARAM_DATA *)mesh->parametric->data;
  DOF_UCHAR_VEC       *touched_coords = data->touched_coords;
  static const REAL_B lambda[3] = { {0.25, 0.25, 0.5, 0.0}, 
				    {0.75, 0.25, 0.0, 0.0},
				    {0.25, 0.75, 0.0, 0.0} };

  GET_DOF_VEC(vec, drdv);
  el = list->el_info.el;
  n_proj = data->n_proj;

  node_v = mesh->node[VERTEX];        
  node_e = mesh->node[EDGE]; 
  n0_v   = drdv->fe_space->admin->n0_dof[VERTEX];
  n0_e   = drdv->fe_space->admin->n0_dof[EDGE];

/****************************************************************************/
/* Step 1: Initialize the parametric element.                               */
/****************************************************************************/
  param_init_element(&list->el_info, mesh->parametric);

/****************************************************************************/
/* Step 2: Now we need to determine three points: The midpoint on the edge  */
/* between the two children and the two midpoints of the children along the */
/* current refinement edge. These three points correspond to the three new  */
/* DOFS that were created.                                                  */
/* If strategy is 0 or 1 we use the (nonlinear) barycentric coords.         */
/* For strategy 2 we use geometric midpoints.                               */
/****************************************************************************/
  if(mesh->parametric->not_all <= 1)
    param_coord_to_world(&list->el_info, nil, 3, lambda, x);
  else {
    cdof_e[2] = el->dof[node_e+2][n0_e];
    for(i = 0; i < N_EDGES_2D; i++)
      cdof_v[i] = el->dof[node_v+i][n0_v];

    for(i = 0; i < DIM_OF_WORLD; i++) {
      x[0][i] = 0.5 * (vec[cdof_v[2]][i] + vec[cdof_e[2]][i]);
      x[1][i] = 0.5 * (vec[cdof_v[0]][i] + vec[cdof_e[2]][i]);
      x[2][i] = 0.5 * (vec[cdof_v[1]][i] + vec[cdof_e[2]][i]);
    }
  }
    
/****************************************************************************/
/* Step 3: We check if any projections need to be done on these three       */
/* points. While doing this, we keep track of touched coordinates.          */
/****************************************************************************/
  cdof[0] = el->child[0]->dof[node_e+1][n0_e];
  cdof[1] = el->child[0]->dof[node_e+0][n0_e];
  cdof[2] = el->child[1]->dof[node_e+1][n0_e];

  if(touched_coords) {
    touched_coords->vec[cdof[0]] = 0;
    touched_coords->vec[cdof[1]] = 0;
    touched_coords->vec[cdof[2]] = 0;
  }

  if(n_proj && n_proj->func) {
    if(list->el_info.projections[0] == n_proj) {
      (n_proj->func)(x[0], &list->el_info, lambda[0]);

      if(touched_coords)
	touched_coords->vec[cdof[0]] = 1;
    }
    if(list->el_info.active_projection == n_proj) {
      (n_proj->func)(x[1], &list->el_info, lambda[1]);
      (n_proj->func)(x[2], &list->el_info, lambda[2]);

      if(touched_coords) {
	touched_coords->vec[cdof[1]] = 1;
	touched_coords->vec[cdof[2]] = 1;
      }
    }
  }

/****************************************************************************/
/* Step 4: Now that any projection has taken place, copy the values stored  */
/* in x[i] to the DOF_REAL_D_VEC.                                           */
/****************************************************************************/
  for(i = 0; i < 3; i++)
    for (j = 0; j < DIM_OF_WORLD; j++)
      vec[cdof[i]][j] = x[i][j];

/****************************************************************************/
/* Step 5: We hand down the data corresponding to the parent edge vertex    */
/* in the refinement edge (its DOF could be deleted!). This concerns coords */
/* and touched coords.                                                      */
/****************************************************************************/
  pdof = el->dof[node_e+2][n0_e];
  cdof[0] = el->child[0]->dof[node_v+2][n0_v];

  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[cdof[0]][j] = vec[pdof][j];

  if(touched_coords)
    touched_coords->vec[cdof[0]] = touched_coords->vec[pdof];

/****************************************************************************/
/* Step 6: Correct the children, if not all elements are to be parametric.  */
/* If the parent element was already affine, then the children will already */
/* have the correct coordinates.                                            */
/****************************************************************************/
  if(touched_coords && !data->i_am_affine) {
    for(i = 0; i < 2; i++) {
      result = 0;
      
      for(j = 0; j < N_EDGES_2D; j++) {
	cdof[0] = el->child[i]->dof[node_e+j][n0_e];
	
	if(touched_coords->vec[cdof[0]]) {
	  result = 1;
	  break;
	}
      }
      if(result == 0) {
	for(j = 0; j < N_EDGES_2D; j++) {
	  cdof_e[j] = el->child[i]->dof[node_e+j][n0_e];
	  cdof_v[j] = el->child[i]->dof[node_v+j][n0_v];
	}
	
	for (j = 0; j < DIM_OF_WORLD; j++) {
	  vec[cdof_e[0]][j] = 0.5*(vec[cdof_v[1]][j] + vec[cdof_v[2]][j]);
	  vec[cdof_e[1]][j] = 0.5*(vec[cdof_v[0]][j] + vec[cdof_v[2]][j]);
	  vec[cdof_e[2]][j] = 0.5*(vec[cdof_v[0]][j] + vec[cdof_v[1]][j]);
	}
      }
    }
  }

/*--------------------------------------------------------------------------*/
/*---  Take care of the neighbour element.                               ---*/
/*--------------------------------------------------------------------------*/
  if (n > 1) {
    el = list[1].el_info.el;

/****************************************************************************/
/* Step 1: Initialize the neighbour element.                                */
/****************************************************************************/
    param_init_element(&list[1].el_info, mesh->parametric);

/****************************************************************************/
/* Step 2: Determine the location of the midpoint on the edge between the   */
/* two children.                                                            */
/****************************************************************************/
    if(mesh->parametric->not_all <= 1)
      param_coord_to_world(&list[1].el_info, nil, 1, lambda, x);
    else {
      cdof_e[2] = el->dof[node_e+2][n0_e];
      cdof_v[2] = el->dof[node_v+2][n0_v];

      for(i = 0; i < DIM_OF_WORLD; i++)
	x[0][i] = 0.5 * (vec[cdof_v[2]][i] + vec[cdof_e[2]][i]);
    }

/****************************************************************************/
/* Step 3: We check if any projections need to be done on this point.       */
/* While doing this, we keep track of touched coordinates.                  */
/****************************************************************************/
    cdof[0] = el->child[0]->dof[node_e+1][n0_e];

    if(touched_coords)
      touched_coords->vec[cdof[0]] = 0;
    
    if(n_proj && n_proj->func) {
      if(list[1].el_info.projections[0] == n_proj) {
	(n_proj->func)(x[0], &list[1].el_info, lambda[0]);
      
	if(touched_coords)
	  touched_coords->vec[cdof[0]] = 1;
      }
    }
    
/****************************************************************************/
/* Step 4: Now that any projection has taken place, copy the values stored  */
/* in x[0] to the DOF_REAL_D_VEC.                                           */
/****************************************************************************/
    for (j = 0; j < DIM_OF_WORLD; j++)
      vec[cdof[0]][j] = x[0][j];

/****************************************************************************/
/* Step 5: Correct the children.                                            */
/****************************************************************************/
    if(touched_coords && !data->i_am_affine) {
      for(i = 0; i < 2; i++) {
	result = 0;
	
	for(j = 0; j < N_EDGES_2D; j++) {
	  cdof[0] = el->child[i]->dof[node_e+j][n0_e];
	  
	  if(touched_coords->vec[cdof[0]]) {
	    result = 1;
	    break;
	}
	}
	if(result == 0) {
	  for(j = 0; j < N_EDGES_2D; j++) {
	    cdof_e[j] = el->child[i]->dof[node_e+j][n0_e];
	    cdof_v[j] = el->child[i]->dof[node_v+j][n0_v];
	  }
	  
	  for (j = 0; j < DIM_OF_WORLD; j++) {
	    vec[cdof_e[0]][j] = 0.5*(vec[cdof_v[1]][j] + vec[cdof_v[2]][j]);
	    vec[cdof_e[1]][j] = 0.5*(vec[cdof_v[0]][j] + vec[cdof_v[2]][j]);
	    vec[cdof_e[2]][j] = 0.5*(vec[cdof_v[0]][j] + vec[cdof_v[1]][j]);
	  }
	}
      }
    }
  }

  return;
}


static void coarse_interpol2_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list, int n)
{
  FUNCNAME("coarse_interpol2_2d");
  EL                  *el;
  REAL_D              *vec = nil;
  int                  node_v, node_e, n0_v, n0_e, j;
  DOF                  cdof, pdof;
  const MESH          *mesh = drdv->fe_space->mesh;
  LAGRANGE_PARAM_DATA *data = (LAGRANGE_PARAM_DATA *)mesh->parametric->data;
  DOF_UCHAR_VEC       *touched_coords = data->touched_coords;

  if (n < 1) return;

  GET_DOF_VEC(vec, drdv);
  el = list->el_info.el;

  node_v = mesh->node[VERTEX];        
  node_e = mesh->node[EDGE]; 
  n0_v   = drdv->fe_space->admin->n0_dof[VERTEX];
  n0_e   = drdv->fe_space->admin->n0_dof[EDGE];

/****************************************************************************/
/*  copy values at refinement vertex to the parent refinement edge.         */
/****************************************************************************/

  cdof = el->child[0]->dof[node_v+2][n0_v];      /* newest vertex is dim */
  pdof = el->dof[node_e+2][n0_e];

  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[pdof][j] = vec[cdof][j];

  if(touched_coords)
    touched_coords->vec[pdof] = touched_coords->vec[cdof];

  return;
}

static void fill_coords2_2d(LAGRANGE_PARAM_DATA *data)
{
  DOF_REAL_D_VEC  *coords = data->coords;
  DOF_UCHAR_VEC   *touched_coords = data->touched_coords;
  NODE_PROJECTION *n_proj = data->n_proj, *active_proj;
  TRAVERSE_STACK  *stack = get_traverse_stack();
  const EL_INFO   *el_info;
  FLAGS            fill_flag = CALL_LEAF_EL|FILL_COORDS|FILL_PROJECTION;
  const DOF     *(*get_dof)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;
  int              i, n;

  get_dof = coords->fe_space->bas_fcts->get_dof_indices;
  admin   = coords->fe_space->admin;

  el_info = traverse_first(stack, coords->fe_space->mesh, -1, fill_flag);
  while (el_info) {
    const DOF *dof = (*get_dof)(el_info->el, admin, nil);
    REAL      *vec, *vec0, *vec1;

    for (i = 0; i < N_VERTICES_2D; i++) {
      vec = coords->vec[dof[i]];
      for (n = 0; n < DIM_OF_WORLD; n++)
	vec[n] = el_info->coord[i][n];

      /* Look for a projection function that applies to vertex[i]. */
      /* Apply this projection if found.                           */

      if(touched_coords)
	touched_coords->vec[dof[i]] = 0;

      if(n_proj && n_proj->func) {
	active_proj = el_info->projections[1 + (i+1)%3];

	if(!active_proj) active_proj = el_info->projections[1 + (i+2)%3];
	if(!active_proj) active_proj = el_info->projections[0];

	if(active_proj == n_proj) {
	  (n_proj->func)(vec, el_info, bary2_2d[i]);

	  if(touched_coords)
	    touched_coords->vec[dof[i]] = 1;
	}
      }
    }

    for (i = 0; i < N_EDGES_2D; i++) {
      int  *voe = vertex_of_edge_2d[i];

      vec = coords->vec[dof[N_VERTICES_2D+i]];
      vec0 = coords->vec[dof[voe[0]]];
      vec1 = coords->vec[dof[voe[1]]];

      for (n = 0; n < DIM_OF_WORLD; n++)
	vec[n] = 0.5*(vec0[n] + vec1[n]);

      /* Look for a projection function that applies to edge[i].   */
      /* Apply this projection if found.                           */

      if(touched_coords)
	touched_coords->vec[dof[N_VERTICES_2D + i]] = 0;

      if(n_proj && n_proj->func) {
	active_proj = el_info->projections[1 + i];

	if(!active_proj) active_proj = el_info->projections[0];

	if(active_proj == n_proj) {
	  (n_proj->func)(vec, el_info, bary2_2d[N_VERTICES_2D + i]);

	  if(touched_coords)
	    touched_coords->vec[dof[N_VERTICES_2D + i]] = 1;
	}
      }
    }

    el_info = traverse_next(stack, el_info);
  }
  free_traverse_stack(stack);

  return;
}


static PARAMETRIC lagrange_parametric1_2d = 
  {"2D Lagrange parametric elements of degree 1",
   param_init_element,
   param_coord_to_world,
   nil,
   det1_2d,
   grd_lambda1_2d,
   false, false, nil};

static PARAMETRIC lagrange_parametric2_2d = 
  {"2D Lagrange parametric elements of degree 2",
    param_init_element,
   param_coord_to_world,
   nil,
   dety_2d,
   grd_lambday_2d,
   false, false, nil};
