/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: file.c,v 1.6 94/07/13 14:22:28 nau Exp $";

/* file save, load, merge ... routines
 */

#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <time.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "global.h"

#include "control.h"
#include "cursor.h"
#include "create.h"
#include "data.h"
#include "dialog.h"
#include "error.h"
#include "file.h"
#include "memory.h"
#include "misc.h"
#include "parse_l.h"
#include "remove.h"
#include "transform.h"

/* ---------------------------------------------------------------------------
 * some local defines
 */
#define	EMERGENCY_NAME		"/tmp/PCB.%i.save"
#define	BACKUP_NAME			"/tmp/PCB.%i.backup"

#if !defined(HAS_ATEXIT) && !defined(HAS_ON_EXIT)
/* ---------------------------------------------------------------------------
 * some local identifiers for machines without an atexit() or on_exit()
 * call
 */
static	char	TMPFilename[80];
#endif

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	WritePCBInfoHeader(FILE *);
static	void	WritePCBDataHeader(FILE *);
static	void	WritePCBFontData(FILE *);
static	void	WritePCBViaData(FILE *);
static	void	WritePCBElementData(FILE *);
static	void	WritePCBLayerData(FILE *, Cardinal);
static	int		WritePCB(FILE *);
static	int		WritePCBFile(char *);
static	int		WritePCBPipe(char *);

/* ---------------------------------------------------------------------------
 * save PCB
 */
int SavePCB(char *Filename)
{
	int		retcode;
	char	*ptr;

	if (!(retcode = WritePCBPipe(Filename)))
	{
			/* setup a new filename */
		ptr = MyStrdup(Filename, "SavePCB()");
		RENAME(PCB->Filename, ptr);
		SetChangedFlag(False);
	}
	return(retcode);
}

/* ---------------------------------------------------------------------------
 * load one element into 'MarkedElement' which may then be positioned
 * parse the file with disabled 'PCB mode' (see parser)
 * if successful, update some other stuff and reposition the loaded element
 */
int LoadMarkedElement(char *Filename)
{
	ElementType		newElement;

		/* clear structure, needed for the routines which release the memory */
	memset(&newElement, 0, sizeof(ElementType));

	if (!ParseElement(&newElement, Filename))
	{
			/* reposition the upper left corner to (0,0) */
		SetElementInfo(&newElement, 1);
		MoveElement(&newElement, -newElement.Rect.X, -newElement.Rect.Y);
		FreeElementMemory(&MyCursor.MarkedElement.Element);
		MyCursor.MarkedElement.Element = newElement;
		MyCursor.MarkedElement.Valid = True;

			/* update cursor display mode */
		HideCursor();
		MyCursor.Mode = MODE_ELEMENT;
		RestoreCursor();
		SetElementnameField();
		SetStatusLine();
		return(0);
	}
		/* disable any old elements that might be loaded */
	MyCursor.MarkedElement.Valid = False;
	return(1);
}

/* ---------------------------------------------------------------------------
 * load PCB
 * parse the file with enabled 'PCB mode' (see parser)
 * if successful, update some other stuff
 */
int LoadPCB(char *Filename)
{
	PCBTypePtr	newPCB = CreateNewPCB();
	int			save;

	if (!ParsePCB(newPCB, Filename))
	{
		RemovePCB(PCB);
		PCB = newPCB;
		SetElementInfo(PCB->Element, PCB->ElementN);
		MyCursor.X = PCB->CursorX;
		MyCursor.Y = PCB->CursorY;

			/* update zoom information. Presetting to a not allowed
			 * value makes sure that the routine is forced to
			 * initialize some things
			 * SetZoom() also calls CenterDisplay()
			 */
		save = PCB->Zoom;
		PCB->Zoom = -1;
		SetZoom(save);

			/* create default font if necessary */
		if (!PCB->Font.Valid)
			CreateDefaultFont();

			/* clear 'changed flag' */
		SetChangedFlag(False);
		PCB->Filename = MyStrdup(Filename, "LoadPCB()");
		UpdateSettingsOnScreen();
		return(0);
	}
	RemovePCB(newPCB);
	return(1);
}

/* ---------------------------------------------------------------------------
 * writes layout header information
 */
static void WritePCBInfoHeader(FILE *FP)
{
	struct passwd	*pwentry;
	struct hostent	*hostentry;
	time_t			currenttime;
	char			hostname[256];

		/* write some useful comments */
	currenttime = time(NULL);
	pwentry = getpwuid(getuid());
	fprintf(FP, "# date: %s", asctime(localtime(&currenttime)));
	fprintf(FP, "# user: %s (%s)\n",
		pwentry->pw_name, pwentry->pw_gecos);
	if (gethostname(hostname, 255) != -1)
	{
		hostentry = gethostbyname(hostname); 
		fprintf(FP, "# host: %s\n", hostentry ? hostentry->h_name : hostname);
	}
}

/* ---------------------------------------------------------------------------
 * writes data header
 */
static void WritePCBDataHeader(FILE *FP)
{
	Cardinal	group,
				entry;

	fprintf(FP, "\nPCB(\"%s\")\n\n", EMPTY(PCB->Name)); 
	fprintf(FP, "Grid(%i %i %i)\n", PCB->Grid, PCB->GridOffsetX, PCB->GridOffsetY);
	fprintf(FP, "Cursor(%i %i %i)\n",
		(int) MyCursor.X, (int) MyCursor.Y, PCB->Zoom);
	fprintf(FP, "Flags(0x%04x)\n", PCB->Flags);
	fputs("Groups(\"", FP);
	for (group = 0; group < MAX_LAYER; group++)
	{
		if (!PCB->LayerGroups.Number[group])
			continue;
		for (entry = 0; entry < PCB->LayerGroups.Number[group]; entry++)
			fprintf(FP, "%d%s", PCB->LayerGroups.Entries[group][entry]+1,
				entry == PCB->LayerGroups.Number[group]-1 ? "" : ",");
		if (group != MAX_LAYER-1)
			fputc(':', FP);
	}
	fputs("\")\n\n", FP);
}

/* ---------------------------------------------------------------------------
 * writes font data of non empty symbols
 */
static void WritePCBFontData(FILE *FP)
{
	Cardinal	i, j;
	LineTypePtr	line;
	FontTypePtr	font;

	for (font = &PCB->Font, i = 0; i <= MAX_FONTPOSITION; i++)
	{
		if (!font->Symbol[i].Valid)
			continue;

		if (isprint(i))
			fprintf(FP, "Symbol('%c' %i)\n(\n", (char) i, (int) font->Symbol[i].Delta);
		else
			fprintf(FP, "Symbol(%i %i)\n(\n", i, (int) font->Symbol[i].Delta);
		for (line = font->Symbol[i].Line, j = font->Symbol[i].LineN; j; j --, line++)
			fprintf(FP, "\tSymbolLine(%i %i %i %i %i)\n",
				(int) line->X1, (int) line->Y1,
				(int) line->X2, (int) line->Y2,
				(int) line->Thickness);
		fputs(")\n", FP);
	}
}

/* ---------------------------------------------------------------------------
 * writes via data
 */
static void WritePCBViaData(FILE *FP)
{
	Cardinal	i;
	PinTypePtr	via;

		/* write information about vias */
	for (via = PCB->Via, i = PCB->ViaN; i; i--, via++)
		fprintf(FP, "Via(%i %i %i \"%s\" 0x%04x)\n",
			(int) via->X, (int) via->Y, (int) via->Thickness,
				EMPTY(via->Name), (int) via->Flags);

}

/* ---------------------------------------------------------------------------
 * writes element data
 */
static void WritePCBElementData(FILE *FP)
{
	ElementTypePtr	element;
	LineTypePtr		line;
	PinTypePtr		pin;
	ArcTypePtr		arc;
	MarkTypePtr		mark;
	Cardinal		i, j;

	for (element = PCB->Element, i = PCB->ElementN; i; i--, element++)
	{
			/* only non empty elements */
		if (!element->LineN && !element->PinN &&
			!element->ArcN && !element->MarkN)
			continue;

		fprintf(FP, "\nElement(\"%s\" \"%s\" %i %i %i)\n(\n",
			EMPTY(element->CanonicalName), EMPTY(element->NameOnPCB),
			(int) element->TextX, (int) element->TextY, (int) element->Direction);

		for (pin = element->Pin, j = element->PinN; j; j--, pin++)
			fprintf(FP, "\tPin(%i %i %i \"%s\" 0x%04x)\n",
				(int) pin->X, (int) pin->Y, (int) pin->Thickness,
				EMPTY(pin->Name), (int) pin->Flags);

		for (line = element->Line, j = element->LineN; j; j--, line++)
			fprintf(FP, "\tElementLine (%i %i %i %i %i)\n",
				(int) line->X1, (int) line->Y1,
				(int) line->X2, (int) line->Y2,
				(int) line->Thickness);

		for (arc = element->Arc, j = element->ArcN; j; j--, arc++)
			fprintf(FP, "\tElementArc (%i %i %i %i %i %i %i)\n",
				(int) arc->X, (int) arc->Y,
				(int) arc->Width, (int) arc->Height,
				(int) arc->StartAngle, (int) arc->Delta,
				(int) arc->Thickness);

		for (mark = element->Mark, j = element->MarkN; j; j--, mark++)
			fprintf(FP, "\tMark (%i %i)\n",
				(int) mark->X, (int) mark->Y);

		fputs(")\n", FP);
	}

}

/* ---------------------------------------------------------------------------
 * writes layer data
 */
static void WritePCBLayerData(FILE *FP, Cardinal Number)
{
	LineTypePtr	line;
	RectTypePtr	rect;
	TextTypePtr	text;
	Cardinal	i;

		/* write information about non empty layers */
	if (!PCB->Layer[Number].LineN &&
		!PCB->Layer[Number].RectN &&
		!PCB->Layer[Number].TextN &&
		(!PCB->Layer[Number].Name || ! *PCB->Layer[Number].Name))
		return;

	fprintf(FP, "Layer (%i \"%s\")\n(\n", (int) Number+1, EMPTY(PCB->Layer[Number].Name));

	for (line = PCB->Layer[Number].Line, i = PCB->Layer[Number].LineN; i; i --, line++)
		fprintf(FP, "\tLine(%i %i %i %i %i 0x%04x)\n",
			(int) line->X1, (int) line->Y1,
			(int) line->X2, (int) line->Y2,
			(int) line->Thickness, (int) line->Flags);

	for (rect = PCB->Layer[Number].Rect, i = PCB->Layer[Number].RectN; i; i --, rect++)
		fprintf(FP, "\tRectangle(%i %i %i %i 0x%04x)\n",
			(int) rect->X, (int) rect->Y,
			(int) rect->Width,(int) rect->Height, (int) rect->Flags);

	for (text = PCB->Layer[Number].Text, i = PCB->Layer[Number].TextN; i; i --, text++)
		fprintf(FP, "\tText(%i %i %i \"%s\" 0x%04x)\n",
			(int) text->X, (int) text->Y, (int) text->Direction,
			EMPTY(text->TextString), (int) text->Flags);

	fputs(")\n", FP);
}

/* ---------------------------------------------------------------------------
 * writes PCB to file
 */
static int WritePCB(FILE *FP)
{
	Cardinal	i;

	WritePCBInfoHeader(FP);
	WritePCBDataHeader(FP);
	WritePCBFontData(FP);
	WritePCBViaData(FP);
	WritePCBElementData(FP);
	for (i = 0; i < MAX_LAYER; i++)
		WritePCBLayerData(FP, i);

	return(STATUS_OK);
}

/* ---------------------------------------------------------------------------
 * writes PCB to file
 */
static int WritePCBFile(char *Filename)
{
	FILE	*fp;
	int		result;

	if ((fp = fopen(Filename, "w")) == NULL)
	{
		MyWarningDialog("can't open file '%s' for writing\n", Filename);
		return(STATUS_ERROR);
	}
	result = WritePCB(fp);
	fclose(fp);
	return(result);
}

/* ---------------------------------------------------------------------------
 * writes PCB to pipe using the command defined by Settings.SaveCommand
 * %f, %s are replaced by the passed filename
 */
static int WritePCBPipe(char *Filename)
{
			FILE				*fp;
			int					result;
	static  DynamicStringType	command;
			char				*p;

		/* clear old string */
	if (command.Data)
		command.Data[0] = '\0';

	for (p = Settings.SaveCommand; *p; p++)
	{
			/* copy character if not special or add string to command */
		if (!(*p == '%' && (*(p+1) == 's' || *(p+1) == 'f' )))
			DSAddCharacter(&command, *p);
		else
		{
			DSAddString(&command, Filename);

				/* skip the character */
			p++;
		}
	}
	DSAddCharacter(&command, '\0');

	if ((fp = popen(command.Data, "w")) == NULL)
	{
		MyWarningDialog("can't execute command '%s'\n", command.Data);
		return(STATUS_ERROR);
	}
	result = WritePCB(fp);
	return(pclose(fp) ? STATUS_ERROR : result);
}

#if !defined(HAS_ATEXIT) && defined(HAS_ON_EXIT)
/* ---------------------------------------------------------------------------
 * just a glue function for systems with on_exit()
 */
void GlueEmergencySave(int status, caddr_t arg)
{
	EmergencySave();
}
#endif

/* ---------------------------------------------------------------------------
 * saves the layout in a temporary file
 * this is used for fatal errors and does not call the program specified
 * in 'saveCommand' for savety reasons
 */
void EmergencySave(void)
{
	char	filename[80];

	if (PCB->Changed)
	{
		sprintf(filename, EMERGENCY_NAME, getpid());
		MyWarning("trying to save your layout in '%s'\n", filename);
		WritePCBFile(filename);
	}
}

/* ---------------------------------------------------------------------------
 * creates a copy of the current PCB data
 */
void Backup(void)
{
	char	filename[80];

	StopBackupAlarm();
	sprintf(filename, BACKUP_NAME, getpid());
	WritePCBFile(filename);
	ContinueBackupAlarm();
}

#if !defined(HAS_ATEXIT) && !defined(HAS_ON_EXIT)
/* ---------------------------------------------------------------------------
 * makes a temporary copy of the data. This is useful for systems which
 * doesn't support calling functions on exit. We use this to save the data
 * before LEX and YACC functions are called because they are able to abort
 * the program.
 */
void SaveTMPData(void)
{
	sprintf(TMPFilename, EMERGENCY_NAME, getpid());
	WritePCBFile(TMPFilename);
}

/* ---------------------------------------------------------------------------
 * removes the temporary copy of the data file
 */
void RemoveTMPData(void)
{
	unlink(TMPFilename);
}
#endif
