/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
* This file is part of the libgltf project.
*
* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/.
*/

#include "libgltf.h"
#include "RenderScene.h"
#include <iostream>

namespace libgltf
{

glTFHandle* gltf_renderer_init(const std::string& jsonfile, std::vector<glTFFile>& o_glTFFiles)
{
    RenderScene* renderScene = new RenderScene();
    bool bRet = renderScene->initScene(jsonfile, o_glTFFiles);
    if( bRet )
    {
        glTFHandle* handle = new glTFHandle;
        handle->renderer = renderScene;
        return handle;
    }
    else
    {
        delete renderScene;
        return 0;
    }
}

int gltf_renderer_set_content(glTFHandle* handle, const std::vector<glTFFile>& inputFiles)
{
    if( !handle )
    {
        std::cerr << "warning: gltf_renderer_set_content() was called with an invalid handle" << std::endl;
        return LIBGLTF_UNKNOWN_ERROR;
    }

    RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
    return renderScene->initRender(inputFiles);
}

void gltf_render_FPS_enable(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->initFPS();
    }
    else
    {
        std::cerr << "warning: gltf_render_FPS_enable() was called with an invalid handle" << std::endl;
    }
}


void gltf_render_FPS_disable(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->endFPS();
    }
    else
    {
        std::cerr << "warning: gltf_render_FPS_disable() was called with an invalid handle" << std::endl;
    }
}

int gltf_prepare_renderer(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->prepareRender(&handle->viewport);
    }
    else
    {
        std::cerr << "warning: gltf_prepare_renderer() was called with an invalid handle" << std::endl;
        return LIBGLTF_UNKNOWN_ERROR;
    }
}

void gltf_renderer(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->render();
    }
    else
    {
        std::cerr << "warning: gltf_renderer() was called with an invalid handle" << std::endl;
    }
}

void gltf_complete_renderer(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->completeRender();
    }
    else
    {
        std::cerr << "warning: gltf_complete_renderer() was called with an invalid handle" << std::endl;
    }
}

void gltf_renderer_release(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->releaseRender();
        delete renderScene;
        delete handle;
    }
    else
    {
        std::cerr << "warning: gltf_renderer_release() was called with an invalid handle" << std::endl;
    }
}

glm::vec3* gltf_get_model_center_pos(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getModelCenterPos();
    }
    else
    {
        std::cerr << "warning: gltf_get_model_center_pos() was called with an invalid handle" << std::endl;
        return 0;
    }
}

void gltf_get_camera_pos(glTFHandle* handle, glm::vec3* eye,
                         glm::vec3* view, glm::vec3* up)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);

        if( eye && view && up )
        {
            renderScene->getCameraPos(eye, view, up);
        }
        else
        {
            std::cerr << "warning: gltf_get_camera_pos() was called with null input parameter" << std::endl;
        }
    }
    else
    {
        std::cerr << "warning: gltf_get_camera_pos() was called with an invalid handle" << std::endl;
        *eye = glm::vec3(0.0f);
        *view = glm::vec3(0.0f);
        *up = glm::vec3(0.0f);
    }
}

double gltf_get_model_size(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getModelSize();
    }
    else
    {
        std::cerr << "warning: gltf_get_model_size() was called with an invalid handle" << std::endl;
        return 0.0;
    }
}

void gltf_renderer_move_camera(glTFHandle* handle, double x, double y,
                               double z, double time)
{
    if(!handle)
    {
        std::cerr << "warning: gltf_renderer_move_camera() was called with an invalid handle" << std::endl;
    }
    else if (time < 0.0)
    {
        std::cerr << "warning: gltf_renderer_move_camera() was called with an invalid time value: " << time << std::endl;
    }
    else
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->getCamera().moveCamera(x, y, z, time);
    }
}

void gltf_renderer_rotate_model(glTFHandle* handle, double horizontal,
                                double vertical, double planar)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        if( renderScene->isInAerialView() && renderScene->isRotationEnabled() )
            renderScene->getCamera().rotateObjectMouse(horizontal, vertical, planar);
    }
    else
    {
        std::cerr << "warning: gltf_renderer_rotate_model() was called with an invalid handle" << std::endl;
    }
}

void gltf_renderer_rotate_camera(glTFHandle* handle, double horizontal,
                                 double vertical, double planar)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        if( renderScene->isRotationEnabled() )
        {
            renderScene->getCamera().rotateCamera(horizontal, vertical, planar);
        }
    }
    else
    {
        std::cerr << "warning: gltf_renderer_rotate_camera() was called with an invalid handle" << std::endl;
    }
}

bool gltf_renderer_fly_camera(glTFHandle* handle, glm::mat4 glPosViewMatrix,
                              double time)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->renderFlyCamera(glPosViewMatrix, time);
    }
    else
    {
        std::cerr << "warning: gltf_renderer_fly_camera() was called with an invalid handle" << std::endl;
        return false;
    }
}

void gltf_enable_rotation(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->enableRotation();
    }
    else
    {
        std::cerr << "warning: gltf_enable_rotation() was called with an invalid handle" << std::endl;
    }
}

void gltf_disable_rotation(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->disableRotation();
    }
    else
    {
        std::cerr << "warning: gltf_disable_rotation() was called with an invalid handle" << std::endl;
    }
}

void gltf_enable_transparency(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->enableTransparency();
    }
    else
    {
        std::cerr << "warning: gltf_enable_transparency() was called with an invalid handle" << std::endl;
    }
}

void gltf_disable_transparency(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->disableTransparency();
    }
    else
    {
        std::cerr << "warning: gltf_disable_transparency() was called with an invalid handle" << std::endl;
    }
}

int gltf_renderer_get_bitmap(glTFHandle** handle, int size,
                             char* buffer, GLenum format, double time)
{
    if (size <= 0)
        return LIBGLTF_INVALID_SIZE;

    if( !handle || !handle[0] )
    {
        std::cerr << "warning: gltf_renderer_get_bitmap() was called with an invalid handle" << std::endl;
        return LIBGLTF_UNKNOWN_ERROR;
    }

    RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle[0]->renderer);
    int status = renderScene->prepareRenderBitmap(&(handle[0]->viewport));
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
    for (int i = 0; i < size; i++)
    {
        if( !handle[i] )
        {
            std::cerr << "warning: gltf_renderer_get_bitmap() was called with an invalid handle" << std::endl;
            return LIBGLTF_UNKNOWN_ERROR;
        }

        RenderScene* tRenderScene = reinterpret_cast<RenderScene*>(handle[i]->renderer);
        tRenderScene->renderBitmap(time);
    }
    renderScene->completeRenderBitmap(&(handle[0]->viewport),
                                      reinterpret_cast<unsigned char*>(buffer), format);
    return LIBGLTF_SUCCESS;
}

void gltf_animation_start(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->startAnimation();
    }
    else
    {
        std::cerr << "warning: gltf_animation_start() was called with an invalid handle" << std::endl;
    }
}

void gltf_animation_stop(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->stopAnimation();
    }
    else
    {
        std::cerr << "warning: gltf_animation_stop() was called with an invalid handle" << std::endl;
    }
}

void gltf_orbit_mode_start(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->startAerialView();
    }
    else
    {
        std::cerr << "warning: gltf_orbit_mode_start() was called with an invalid handle" << std::endl;
    }
}

void gltf_orbit_mode_stop(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->stopAerialView();
    }
    else
    {
        std::cerr << "warning: gltf_orbit_mode_stop() was called with an invalid handle" << std::endl;
    }
}

bool gltf_animation_is_playing(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->isAnimPlay();
    }
    else
    {
        std::cerr << "warning: gltf_animation_is_playing() was called with an invalid handle" << std::endl;
        return false;
    }
}

void gltf_animation_set_looping(glTFHandle* handle, bool loop)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->setAnimLoop(loop);
    }
    else
    {
        std::cerr << "warning: gltf_animation_set_looping() was called with an invalid handle" << std::endl;
    }
}

bool gltf_animation_get_looping(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getAnimLoop();
    }
    else
    {
        std::cerr << "warning: gltf_animation_get_looping() was called with an invalid handle" << std::endl;
        return false;
    }
}

double gltf_animation_get_duration(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getAnimDuration();
    }
    else
    {
        std::cerr << "warning: gltf_animation_get_duration() was called with an invalid handle" << std::endl;
        return 0.0;
    }
}

void gltf_animation_set_time(glTFHandle* handle, double time)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        if (time <= 0)
        {
            renderScene->setAnimTime(0.0);
        }
        else
        {
            renderScene->setAnimTime(time);
        }
    }
    else
    {
        std::cerr << "warning: gltf_animation_set_time() was called with an invalid handle" << std::endl;
    }
}

double gltf_animation_get_time(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getAnimTime();
    }
    else
    {
        std::cerr << "warning: gltf_animation_get_time() was called with an invalid handle" << std::endl;
        return 0.0;
    }
}

void gltf_animation_resume(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->resumeAnim();
    }
    else
    {
        std::cerr << "warning: gltf_animation_resume() was called with an invalid handle" << std::endl;
    }
}

void gltf_set_view_matrix(glTFHandle* handle, const glm::mat4& viewMatrix)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->setViewMatrix(viewMatrix);
    }
    else
    {
        std::cerr << "warning: gltf_set_view_matrix() was called with an invalid handle" << std::endl;
    }
}

void gltf_get_view_matrix(glTFHandle* handle, glm::mat4& viewMatrix)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        viewMatrix = renderScene->getViewMatrix();
    }
    else
    {
        std::cerr << "warning: gltf_get_view_matrix() was called with an invalid handle" << std::endl;
    }
}

void gltf_reset_view_matrix(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->resetViewMatrix();
    }
    else
    {
        std::cerr << "warning: gltf_reset_view_matrix() was called with an invalid handle" << std::endl;
    }
}

void gltf_start_patrol(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->startPatrol();
    }
    else
    {
        std::cerr << "warning: gltf_start_patrol() was called with an invalid handle" << std::endl;
    }
}

void gltf_end_patrol(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->endPatrol();
    }
    else
    {
        std::cerr << "warning: gltf_end_patrol() was called with an invalid handle" << std::endl;
    }
}

void gltf_enable_MSAA(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->enableMSAA();
    }
    else
    {
        std::cerr << "warning: gltf_enable_MSAA() was called with an invalid handle" << std::endl;
    }
}

void gltf_disable_MSAA(glTFHandle* handle)
{
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->disableMSAA();
    }
    else
    {
        std::cerr << "warning: gltf_disable_MSAA() was called with an invalid handle" << std::endl;
    }
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
