#include <Python.h>
#include <pystate.h>
#include <stdio.h>
#include <float.h>

/**********************************************************************/
/*  Buffer Utility Functions                                          */
/**********************************************************************/

static PyObject *
getBuffer( PyObject *obj) 
{
	if (!obj) return PyErr_Format(PyExc_RuntimeError,
				      "NULL object passed to getBuffer()");
	if (obj->ob_type->tp_as_buffer == NULL) {
		return PyObject_CallMethod(obj, "__buffer__", NULL);
	} else {
		Py_INCREF(obj);  /* Since CallMethod returns a new object when it
				    succeeds, We'll need to DECREF later to free it.
				    INCREF ordinary buffers here so we don't have to
				    remember where the buffer came from at DECREF time.
				 */
		return obj;
	}
}

static int 
isBuffer (PyObject *obj) 
{
  /* Either it defines the buffer API, or it is an instance which returns
     a buffer when obj.__buffer__() is called */
  PyObject *buf = getBuffer(obj);
  int ans = buf->ob_type->tp_as_buffer != NULL;
  Py_DECREF(buf);
  return ans;
}

/**********************************************************************/

static int 
getWriteBufferDataPtr(PyObject *buffobj, void **buff) 
{
  int rval = -1;
  PyObject *buff2;
  if ((buff2 = getBuffer(buffobj)))
  {
	if (buff2->ob_type->tp_as_buffer->bf_getwritebuffer)
		rval = buff2->ob_type->tp_as_buffer->bf_getwritebuffer(buff2, 
									 0, buff);
	Py_DECREF(buff2);
  }
  return rval;
}

/**********************************************************************/

static int 
isBufferWriteable (PyObject *buffobj) 
{
  void *ptr;
  int rval = -1;
  rval = getWriteBufferDataPtr(buffobj, &ptr);
  if (rval == -1)
    PyErr_Clear(); /* Since we're just "testing", it's not really an error */
  return rval != -1;
}

/**********************************************************************/

static int 
getReadBufferDataPtr(PyObject *buffobj, void **buff) 
{
  int rval = -1;
  PyObject *buff2;
  if ((buff2 = getBuffer(buffobj))) {
	  if (buff2->ob_type->tp_as_buffer->bf_getreadbuffer)
		  rval = buff2->ob_type->tp_as_buffer->bf_getreadbuffer(buff2, 
									0, buff);
	  Py_DECREF(buff2);
  }
  return rval;
}

/**********************************************************************/

static int 
getBufferSize(PyObject *buffobj) 
{
  int segcount, size=0;
  PyObject *buff2;
  if ((buff2 = getBuffer(buffobj))) 
    {
      segcount = buff2->ob_type->tp_as_buffer->bf_getsegcount(buff2, 
								&size);
      Py_DECREF(buff2);
    }
  else
    size = -1;
  return size;
}

static long NA_getBufferPtrAndSize(PyObject *buffobj, int readonly, void **ptr)
{
	long rval;
	if (readonly)
		rval = getReadBufferDataPtr(buffobj, ptr);
	else 
		rval = getWriteBufferDataPtr(buffobj, ptr);
	return rval;
}

static int NA_checkOneCBuffer(char *name, long niter, 
		     void *buffer, long bsize, size_t typesize)
{
	Int64 lniter = niter, ltypesize = typesize;

	if (lniter*ltypesize > bsize) {
		PyErr_Format(_Error, 
			     "%s: access out of buffer. niter=%d typesize=%d bsize=%d",
			     name, (int) niter, (int) typesize, (int) bsize);
		return -1;
	}
	if ((typesize <= sizeof(Float64)) && (((long) buffer) % typesize)) {
		PyErr_Format(_Error, 
			     "%s: buffer not aligned on %d byte boundary.",
			     name, (int) typesize);
		return -1;
	}
	return 0;
}

static int NA_checkIo(char *name, 
		      int wantIn, int wantOut, int gotIn, int gotOut)
{
	if (wantIn != gotIn) {
		PyErr_Format(_Error,
		   "%s: wrong # of input buffers. Expected %d.  Got %d.", 
		    name, wantIn, gotIn);
		return -1;
	}
	if (wantOut != gotOut) {
		PyErr_Format(_Error,
		   "%s: wrong # of output buffers. Expected %d.  Got %d.", 
		    name, wantOut, gotOut);
		return -1;
	}
	return 0;
}

static int NA_checkNCBuffers(char *name, int N, long niter, 
			    void **buffers, long *bsizes,
			    Int8 *typesizes, Int8 *iters)
{
	int i;
	for (i=0; i<N; i++)
		if (NA_checkOneCBuffer(name, iters[i] ? iters[i] : niter, 
			      buffers[i], bsizes[i], typesizes[i]))
			return -1;
	return 0;
}

static int NA_checkOneStriding(char *name, long dim, maybelong *shape,
	       long offset, maybelong *stride, long buffersize, long itemsize, 
	       int align)
{
  int i;
  long omin=offset, omax=offset;
  long alignsize = (itemsize <= sizeof(Float64) ? itemsize : sizeof(Float64));
  
  if (align && (offset % alignsize)) {
    PyErr_Format(_Error, 
		 "%s: buffer not aligned on %d byte boundary.",
		 name, (int) alignsize);
    return -1;
  }
  for(i=0; i<dim; i++) {
    long strideN = stride[i] * (shape[i]-1);
    long tmax = omax + strideN;
    long tmin = omin + strideN;
    if (shape[i]-1 >= 0) {  /* Skip dimension == 0. */
      omax = MAX(omax, tmax);
      omin = MIN(omin, tmin);
      if (align && (ABS(stride[i]) % alignsize)) {
	PyErr_Format(_Error, 
         "%s: stride %d not aligned on %d byte boundary.",
	     name, (int) stride[i], (int) alignsize);
	return -1;
      }
      if (omax + itemsize > buffersize) {
	PyErr_Format(_Error, 
           "%s: access beyond buffer. offset=%d buffersize=%d",
		     name, (int) (omax+itemsize-1), (int) buffersize);
	return -1;
      }
      if (omin < 0) {
	PyErr_Format(_Error, 
	     "%s: access before buffer. offset=%d buffersize=%d",
		     name, (int) omin, (int) buffersize);
	return -1;
      }
    }
  }
  return 0;
}

Float64 NA_get_Float64(PyArrayObject *a, long offset)
{
	switch(a->descr->type_num) {
	case tBool:    
		return NA_GETP(a, Bool, (NA_PTR(a)+offset)) != 0;
	case tInt8:    
		return NA_GETP(a, Int8, (NA_PTR(a)+offset));
	case tUInt8:   
		return NA_GETP(a, UInt8, (NA_PTR(a)+offset));
	case tInt16:   
		return NA_GETP(a, Int16, (NA_PTR(a)+offset));
	case tUInt16:  
		return NA_GETP(a, UInt16, (NA_PTR(a)+offset));
	case tInt32:  
		return NA_GETP(a, Int32, (NA_PTR(a)+offset));
	case tUInt32:  
		return NA_GETP(a, UInt32, (NA_PTR(a)+offset));
	case tInt64:  
		return NA_GETP(a, Int64, (NA_PTR(a)+offset));
	#if HAS_UINT64
	case tUInt64:  
		return NA_GETP(a, UInt64, (NA_PTR(a)+offset));
	#endif
	case tFloat32:
		return NA_GETP(a, Float32, (NA_PTR(a)+offset));
	case tFloat64:
		return NA_GETP(a, Float64, (NA_PTR(a)+offset));
	case tComplex32:  /* Since real value is first */
		return NA_GETP(a, Float32, (NA_PTR(a)+offset));
	case tComplex64:  /* Since real value is first */
		return NA_GETP(a, Float64, (NA_PTR(a)+offset));
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unknown type %d in NA_get_Float64",
			      a->descr->type_num); 
	}
	return 0; /* suppress warning */
}

void NA_set_Float64(PyArrayObject *a, long offset, Float64 v)
{
	Bool b;

	switch(a->descr->type_num) {
	case tBool:
		b = (v != 0);
		NA_SETP(a, Bool, (NA_PTR(a)+offset), b);
		break;
	case tInt8:    NA_SETP(a, Int8, (NA_PTR(a)+offset), v);
		break;
	case tUInt8:   NA_SETP(a, UInt8, (NA_PTR(a)+offset), v);
		break;
	case tInt16:   NA_SETP(a, Int16, (NA_PTR(a)+offset), v);
		break;
	case tUInt16:  NA_SETP(a, UInt16, (NA_PTR(a)+offset), v);
		break;
	case tInt32:   NA_SETP(a, Int32, (NA_PTR(a)+offset), v);
		break;
	case tUInt32:   NA_SETP(a, UInt32, (NA_PTR(a)+offset), v);
		break;
	case tInt64:   NA_SETP(a, Int64, (NA_PTR(a)+offset), v);
		break;
        #if HAS_UINT64
	case tUInt64:   NA_SETP(a, UInt64, (NA_PTR(a)+offset), v);
		break;
	#endif
	case tFloat32: 
		NA_SETP(a, Float32, (NA_PTR(a)+offset), v);
		break;
	case tFloat64: 
		NA_SETP(a, Float64, (NA_PTR(a)+offset), v);
		break;
	case tComplex32: {
		NA_SETP(a, Float32, (NA_PTR(a)+offset), v);
		NA_SETP(a, Float32, (NA_PTR(a)+offset+sizeof(Float32)), 0);
		break;
	}
	case tComplex64: {
		NA_SETP(a, Float64, (NA_PTR(a)+offset), v);
		NA_SETP(a, Float64, (NA_PTR(a)+offset+sizeof(Float64)), 0);
		break;
	}
	default:
		PyErr_Format( PyExc_TypeError, 
			      "Unknown type %d in NA_set_Float64",
			      a->descr->type_num ); 
		PyErr_Print();
	}
}

static int 
NA_overflow(PyArrayObject *a, Float64 v)
{
	if ((a->flags & CHECKOVERFLOW) == 0) return 0;

	switch(a->descr->type_num) {
	case tBool:  
		return 0;
	case tInt8:     
		if ((v < -128) || (v > 127))      goto _fail;
		return 0;
	case tUInt8:    
		if ((v < 0) || (v > 255))         goto _fail;
		return 0;
	case tInt16:    
		if ((v < -32768) || (v > 32767))  goto _fail;
		return 0;
	case tUInt16:	
		if ((v < 0) || (v > 65535))       goto _fail;
		return 0;
	case tInt32:   	
		if ((v < -2147483648.) || 
		    (v > 2147483647.))           goto _fail;
		return 0;
	case tUInt32:  	
		if ((v < 0) || (v > 4294967295.)) goto _fail;
		return 0;
	case tInt64: 	
		if ((v < -9223372036854775808.) || 
		    (v > 9223372036854775807.))    goto _fail;
		return 0;
        #if HAS_UINT64
	case tUInt64:	
		if ((v < 0) || 
		    (v > 18446744073709551615.))    goto _fail;
		return 0;
	#endif
	case tFloat32: 
		if ((v < -FLT_MAX) || (v > FLT_MAX)) goto _fail;
		return 0;
	case tFloat64: 
		return 0;
	case tComplex32: 
		if ((v < -FLT_MAX) || (v > FLT_MAX)) goto _fail;
		return 0;
	case tComplex64: 
		return 0;
	default:
		PyErr_Format( PyExc_TypeError, 
			      "Unknown type %d in NA_overflow",
			      a->descr->type_num ); 
		PyErr_Print();
		return -1;
	}
  _fail:
	PyErr_Format(PyExc_OverflowError, "value out of range for array");
	return -1;
}

Complex64 NA_get_Complex64(PyArrayObject *a, long offset)
{
	Complex32 v0;
	Complex64 v;

	switch(a->descr->type_num) {
	case tComplex32: 
		v0 = NA_GETP(a, Complex32, (NA_PTR(a)+offset));
		v.r = v0.r;
		v.i = v0.i;
		break;
	case tComplex64:
		v = NA_GETP(a, Complex64, (NA_PTR(a)+offset));
		break;
	default:
		v.r = NA_get_Float64(a, offset);
		v.i = 0;
		break;
	}
	return v;
}

void NA_set_Complex64(PyArrayObject *a, long offset, Complex64 v)
{
	Complex32 v0;

	switch(a->descr->type_num) {
	case tComplex32:
		v0.r = v.r;
		v0.i = v.i;
		NA_SETP(a, Complex32, (NA_PTR(a)+offset), v0);
		break;
	case tComplex64:
		NA_SETP(a, Complex64, (NA_PTR(a)+offset), v);
		break;
	default:
		NA_set_Float64(a, offset, v.r);
		break;
	}
}

Int64 NA_get_Int64(PyArrayObject *a, long offset)
{
	switch(a->descr->type_num) {
	case tBool:    
		return NA_GETP(a, Bool, (NA_PTR(a)+offset)) != 0;
	case tInt8:    
		return NA_GETP(a, Int8, (NA_PTR(a)+offset));
	case tUInt8:   
		return NA_GETP(a, UInt8, (NA_PTR(a)+offset));
	case tInt16:   
		return NA_GETP(a, Int16, (NA_PTR(a)+offset));
	case tUInt16:  
		return NA_GETP(a, UInt16, (NA_PTR(a)+offset));
	case tInt32:  
		return NA_GETP(a, Int32, (NA_PTR(a)+offset));
	case tUInt32:  
		return NA_GETP(a, UInt32, (NA_PTR(a)+offset));
	case tInt64:  
		return NA_GETP(a, Int64, (NA_PTR(a)+offset));
	case tUInt64:  
		return NA_GETP(a, UInt64, (NA_PTR(a)+offset));
	case tFloat32:
		return NA_GETP(a, Float32, (NA_PTR(a)+offset));
	case tFloat64:
		return NA_GETP(a, Float64, (NA_PTR(a)+offset));
	case tComplex32:
		return NA_GETP(a, Float32, (NA_PTR(a)+offset));
	case tComplex64:
		return NA_GETP(a, Float64, (NA_PTR(a)+offset));
	default:
		PyErr_Format( PyExc_TypeError, 
			      "Unknown type %d in NA_get_Int64",
			      a->descr->type_num); 
		PyErr_Print();
	}
	return 0; /* suppress warning */
}

void NA_set_Int64(PyArrayObject *a, long offset, Int64 v)
{
	Bool b;

	switch(a->descr->type_num) {
	case tBool:
		b = (v != 0);
		NA_SETP(a, Bool, (NA_PTR(a)+offset), b);
		break;
	case tInt8:    NA_SETP(a, Int8, (NA_PTR(a)+offset), v);
		break;
	case tUInt8:   NA_SETP(a, UInt8, (NA_PTR(a)+offset), v);
		break;
	case tInt16:   NA_SETP(a, Int16, (NA_PTR(a)+offset), v);
		break;
	case tUInt16:  NA_SETP(a, UInt16, (NA_PTR(a)+offset), v);
		break;
	case tInt32:   NA_SETP(a, Int32, (NA_PTR(a)+offset), v);
		break;
	case tUInt32:   NA_SETP(a, UInt32, (NA_PTR(a)+offset), v);
		break;
	case tInt64:   NA_SETP(a, Int64, (NA_PTR(a)+offset), v);
		break;
	case tUInt64:   NA_SETP(a, UInt64, (NA_PTR(a)+offset), v);
		break;
	case tFloat32: 
		NA_SETP(a, Float32, (NA_PTR(a)+offset), v);
		break;
	case tFloat64: 
		NA_SETP(a, Float64, (NA_PTR(a)+offset), v);
		break;
	case tComplex32: 
		NA_SETP(a, Float32, (NA_PTR(a)+offset), v);
		NA_SETP(a, Float32, (NA_PTR(a)+offset+sizeof(Float32)), 0);
		break;
	case tComplex64: 
		NA_SETP(a, Float64, (NA_PTR(a)+offset), v);
		NA_SETP(a, Float64, (NA_PTR(a)+offset+sizeof(Float64)), 0);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unknown type %d in NA_set_Int64",
			      a->descr->type_num); 
		PyErr_Print();
	}
}

/*  NA_get_offset computes the offset specified by the set of indices.
If N > 0, the indices are taken from the outer dimensions of the array.
If N < 0, the indices are taken from the inner dimensions of the array.
If N == 0, the offset is 0.
*/
long NA_get_offset(PyArrayObject *a, int N, ...)
{
	int i;
	long offset = 0;
	va_list ap;
	va_start(ap, N);
	if (N > 0) { /* compute offset of "outer" indices. */
		for(i=0; i<N; i++)
			offset += va_arg(ap, long) * a->strides[i];
	} else {   /* compute offset of "inner" indices. */
		N = -N;
		for(i=0; i<N; i++)
			offset += va_arg(ap, long) * a->strides[a->nd-N+i];
	}
	va_end(ap);
	return offset;
}

Float64 NA_get1_Float64(PyArrayObject *a, long i)
{
	long offset = i * a->strides[0];
	return NA_get_Float64(a, offset);
}

Float64 NA_get2_Float64(PyArrayObject *a, long i, long j)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1];
	return NA_get_Float64(a, offset);
}

Float64 NA_get3_Float64(PyArrayObject *a, long i, long j, long k)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1] 
		+ k * a->strides[2];
	return NA_get_Float64(a, offset);
}

void NA_set1_Float64(PyArrayObject *a, long i, Float64 v)
{
	long offset = i * a->strides[0];
	NA_set_Float64(a, offset, v);
}

void NA_set2_Float64(PyArrayObject *a, long i, long j, Float64 v)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1];
	NA_set_Float64(a, offset, v);
}

void NA_set3_Float64(PyArrayObject *a, long i, long j, long k, Float64 v)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1] 
		+ k * a->strides[2];
	NA_set_Float64(a, offset, v);
}

Complex64 NA_get1_Complex64(PyArrayObject *a, long i)
{
	long offset = i * a->strides[0];
	return NA_get_Complex64(a, offset);
}

Complex64 NA_get2_Complex64(PyArrayObject *a, long i, long j)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1];
	return NA_get_Complex64(a, offset);
}

Complex64 NA_get3_Complex64(PyArrayObject *a, long i, long j, long k)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1] 
		+ k * a->strides[2];
	return NA_get_Complex64(a, offset);
}

void NA_set1_Complex64(PyArrayObject *a, long i, Complex64 v)
{
	long offset = i * a->strides[0];
	NA_set_Complex64(a, offset, v);
}

void NA_set2_Complex64(PyArrayObject *a, long i, long j, Complex64 v)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1];
	NA_set_Complex64(a, offset, v);
}

void NA_set3_Complex64(PyArrayObject *a, long i, long j, long k, Complex64 v)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1] 
		+ k * a->strides[2];
	NA_set_Complex64(a, offset, v);
}

Int64 NA_get1_Int64(PyArrayObject *a, long i)
{
	long offset = i * a->strides[0];
	return NA_get_Int64(a, offset);
}

Int64 NA_get2_Int64(PyArrayObject *a, long i, long j)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1];
	return NA_get_Int64(a, offset);
}

Int64 NA_get3_Int64(PyArrayObject *a, long i, long j, long k)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1] 
		+ k * a->strides[2];
	return NA_get_Int64(a, offset);
}

void NA_set1_Int64(PyArrayObject *a, long i, Int64 v)
{
	long offset = i * a->strides[0];
	NA_set_Int64(a, offset, v);
}

void NA_set2_Int64(PyArrayObject *a, long i, long j, Int64 v)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1];
	NA_set_Int64(a, offset, v);
}

void NA_set3_Int64(PyArrayObject *a, long i, long j, long k, Int64 v)
{
	long offset  = i * a->strides[0] 
		+ j * a->strides[1] 
		+ k * a->strides[2];
	NA_set_Int64(a, offset, v);
}

/* SET_CMPLX could be made faster by factoring it into 3 seperate loops.
*/
#define NA_SET_CMPLX(a, type, base, cnt, in)                                  \
{                                                                             \
        int i;                                                                \
	int stride = a->strides[ a->nd - 1];                                  \
        NA_SET1D(a, type, base, cnt, in);                                     \
	base = NA_PTR(a) + offset + sizeof(type);                             \
	for(i=0; i<cnt; i++) {                                                \
		NA_SETP(a, Float32, base, 0);                                 \
		base += stride;                                               \
	}                                                                     \
}

static int
NA_get1D_Float64(PyArrayObject *a, long offset, int cnt, Float64*out)
{
	char *base = NA_PTR(a) + offset;

	switch(a->descr->type_num) {
	case tBool:
		NA_GET1D(a, Bool, base, cnt, out); 
		break;
	case tInt8:    
		NA_GET1D(a, Int8, base, cnt, out); 
		break;
	case tUInt8:   
		NA_GET1D(a, UInt8, base, cnt, out); 
		break;
	case tInt16:   
		NA_GET1D(a, Int16, base, cnt, out); 
		break;
	case tUInt16:  
		NA_GET1D(a, UInt16, base, cnt, out); 
		break;
	case tInt32:  
		NA_GET1D(a, Int32, base, cnt, out); 
		break;
	case tUInt32:  
		NA_GET1D(a, UInt32, base, cnt, out); 
		break;
	case tInt64:  
		NA_GET1D(a, Int64, base, cnt, out); 
		break;
        #if HAS_UINT64
	case tUInt64:  
		NA_GET1D(a, UInt64, base, cnt, out); 
		break;
        #endif
	case tFloat32: 
		NA_GET1D(a, Float32, base, cnt, out); 
		break;
	case tFloat64: 
		NA_GET1D(a, Float64, base, cnt, out); 
		break;
	case tComplex32:
		NA_GET1D(a, Float32, base, cnt, out);
		break;
	case tComplex64:
		NA_GET1D(a, Float64, base, cnt, out);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unknown type %d in NA_get1D_Float64", 
			      a->descr->type_num); 
		PyErr_Print();
		return -1;
	}
	return 0;
}

static Float64 *
NA_alloc1D_Float64(PyArrayObject *a, long offset, int cnt)
{
	Float64 *result = PyMem_New(Float64, cnt);
	if (!result) return NULL;
	if (NA_get1D_Float64(a, offset, cnt, result) < 0) {
		PyMem_Free(result);
		return NULL;
	}
	return result;
}

static int
NA_set1D_Float64(PyArrayObject *a, long offset, int cnt, Float64*in)
{
	char *base = NA_PTR(a) + offset;

	switch(a->descr->type_num) {
	case tBool:
		NA_SET1D(a, Bool, base, cnt, in); 
		break;
	case tInt8:    
		NA_SET1D(a, Int8, base, cnt, in); 
		break;
	case tUInt8:   
		NA_SET1D(a, UInt8, base, cnt, in); 
		break;
	case tInt16:   
		NA_SET1D(a, Int16, base, cnt, in); 
		break;
	case tUInt16:  
		NA_SET1D(a, UInt16, base, cnt, in); 
		break;
	case tInt32:  
		NA_SET1D(a, Int32, base, cnt, in); 
		break;
	case tUInt32:  
		NA_SET1D(a, UInt32, base, cnt, in); 
		break;
	case tInt64:  
		NA_SET1D(a, Int64, base, cnt, in); 
		break;
        #if HAS_UINT64
	case tUInt64:  
		NA_SET1D(a, UInt64, base, cnt, in); 
		break;
	#endif
	case tFloat32: 
		NA_SET1D(a, Float32, base, cnt, in); 
		break;
	case tFloat64: 
		NA_SET1D(a, Float64, base, cnt, in); 
		break;
	case tComplex32:
		NA_SET_CMPLX(a, Float32, base, cnt, in);
		break;
	case tComplex64:
		NA_SET_CMPLX(a, Float64, base, cnt, in);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unknown type %d in NA_set1D_Float64",
			      a->descr->type_num); 
		PyErr_Print();
		return -1;
	}
	return 0;
}

static int
NA_get1D_Int64(PyArrayObject *a, long offset, int cnt, Int64*out)
{
	char *base = NA_PTR(a) + offset;

	switch(a->descr->type_num) {
	case tBool:
		NA_GET1D(a, Bool, base, cnt, out); 
		break;
	case tInt8:    
		NA_GET1D(a, Int8, base, cnt, out); 
		break;
	case tUInt8:   
		NA_GET1D(a, UInt8, base, cnt, out); 
		break;
	case tInt16:   
		NA_GET1D(a, Int16, base, cnt, out); 
		break;
	case tUInt16:  
		NA_GET1D(a, UInt16, base, cnt, out); 
		break;
	case tInt32:  
		NA_GET1D(a, Int32, base, cnt, out); 
		break;
	case tUInt32:  
		NA_GET1D(a, UInt32, base, cnt, out); 
		break;
	case tInt64:  
		NA_GET1D(a, Int64, base, cnt, out); 
		break;
	case tUInt64:  
		NA_GET1D(a, UInt64, base, cnt, out); 
		break;
	case tFloat32: 
		NA_GET1D(a, Float32, base, cnt, out); 
		break;
	case tFloat64: 
		NA_GET1D(a, Float64, base, cnt, out); 
		break;
	case tComplex32:
		NA_GET1D(a, Float32, base, cnt, out);
		break;
	case tComplex64:
		NA_GET1D(a, Float64, base, cnt, out);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unknown type %d in NA_get1D_Int64",
			      a->descr->type_num); 
		PyErr_Print();
		return -1;
	}
	return 0;
}

static Int64 *
NA_alloc1D_Int64(PyArrayObject *a, long offset, int cnt)
{
	Int64 *result = PyMem_New(Int64, cnt);
	if (!result) return NULL;
	if (NA_get1D_Int64(a, offset, cnt, result) < 0) {
		PyMem_Free(result);
		return NULL;
	}
	return result;
}

static int
NA_set1D_Int64(PyArrayObject *a, long offset, int cnt, Int64*in)
{
	char *base = NA_PTR(a) + offset;

	switch(a->descr->type_num) {
	case tBool:
		NA_SET1D(a, Bool, base, cnt, in); 
		break;
	case tInt8:    
		NA_SET1D(a, Int8, base, cnt, in); 
		break;
	case tUInt8:   
		NA_SET1D(a, UInt8, base, cnt, in); 
		break;
	case tInt16:   
		NA_SET1D(a, Int16, base, cnt, in); 
		break;
	case tUInt16:  
		NA_SET1D(a, UInt16, base, cnt, in); 
		break;
	case tInt32:  
		NA_SET1D(a, Int32, base, cnt, in); 
		break;
	case tUInt32:  
		NA_SET1D(a, UInt32, base, cnt, in); 
		break;
	case tInt64:  
		NA_SET1D(a, Int64, base, cnt, in); 
		break;
	case tUInt64:  
		NA_SET1D(a, UInt64, base, cnt, in); 
		break;
	case tFloat32: 
		NA_SET1D(a, Float32, base, cnt, in); 
		break;
	case tFloat64: 
		NA_SET1D(a, Float64, base, cnt, in); 
		break;
	case tComplex32:
		NA_SET_CMPLX(a, Float32, base, cnt, in);
		break;
	case tComplex64:
		NA_SET_CMPLX(a, Float64, base, cnt, in);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unknown type %d in NA_set1D_Int64",
			      a->descr->type_num); 
		PyErr_Print();
		return -1;
	}
	return 0;
}

static int
NA_get1D_Complex64(PyArrayObject *a, long offset, int cnt, Complex64*out)
{
	char *base = NA_PTR(a) + offset;

	switch(a->descr->type_num) {
	case tComplex64:
		NA_GET1D(a, Complex64, base, cnt, out);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unsupported type %d in NA_get1D_Complex64",
			      a->descr->type_num); 
		PyErr_Print();
		return -1;
	}
	return 0;
}

static int 
NA_set1D_Complex64(PyArrayObject *a, long offset, int cnt, Complex64*in)
{
	char *base = NA_PTR(a) + offset;

	switch(a->descr->type_num) {
	case tComplex64:
		NA_SET1D(a, Complex64, base, cnt, in);
		break;
	default:
		PyErr_Format( PyExc_TypeError,
			      "Unsupported type %d in NA_set1D_Complex64",
			      a->descr->type_num); 
		PyErr_Print();
		return -1;
	}
	return 0;
}

#if LP64
#define PlatBigInt PyInt_FromLong
#define PlatBigUInt PyLong_FromUnsignedLong
#else
#define PlatBigInt PyLong_FromLongLong
#define PlatBigUInt PyLong_FromUnsignedLongLong
#endif

static int 
_checkOffset(PyArrayObject *a, long offset)
{
	long finaloffset = a->byteoffset + offset;
	long size = getBufferSize(a->_data);	
	if (size < 0) {
		PyErr_Format(_Error,
			     "can't get buffer size");
		return -1;
	}
	if (finaloffset < 0 || finaloffset > size) {
		PyErr_Format(_Error,
			     "invalid buffer offset");
		return -1;
	}
	return 0;
}

static PyObject *
NA_getPythonScalar(PyArrayObject *a, long offset)
{
	int type = a->descr->type_num;
	PyObject *rval = NULL;

	if (_checkOffset(a, offset) < 0)
		goto _exit;

	switch(type) {
	case tBool:
        case tInt8:
	case tUInt8:
        case tInt16:
	case tUInt16:
	case tInt32: {
		Int64 v = NA_get_Int64(a, offset);
		rval = PyInt_FromLong(v);
		break;
	}
	case tUInt32: {
		Int64 v = NA_get_Int64(a, offset);
		rval = PlatBigUInt(v); 
		break;
	}
	case tInt64: {
		Int64 v = NA_get_Int64(a, offset);
		rval = PlatBigInt( v);
		break;
	}
	case tUInt64: {
		Int64 v = NA_get_Int64(a, offset);
		rval = PlatBigUInt( v);
		break;
	}
	case tFloat32:
	case tFloat64: {
		Float64 v = NA_get_Float64(a, offset);
		rval = PyFloat_FromDouble( v );
		break;
	}
	case tComplex32:
	case tComplex64: 
	{
		Complex64 v = NA_get_Complex64(a, offset);
		rval = PyComplex_FromDoubles(v.r, v.i);
		break;
	}
	default:
		rval = PyErr_Format(PyExc_TypeError, 
				    "NA_getPythonScalar: bad type %d\n", 
				    type);
	}
  _exit:
	return rval;
}

static int        
_setFromPythonScalarCore(PyArrayObject *a, long offset, PyObject*value, int entries)
{
	Int64 v;
	int rval = -1;

	if (entries >= 100) {
		PyErr_Format(PyExc_RuntimeError, 
			     "NA_setFromPythonScalar: __tonumtype__ conversion chain too long");
		goto _exit;
	} else if (PyInt_Check(value)) {
		v = PyInt_AsLong(value);
		if (NA_overflow(a, v) < 0)
			return -1;
		NA_set_Int64(a, offset, v);
	} else if (PyLong_Check(value)) {  
		if (a->descr->type_num == tInt64) {
			v = (Int64) PyLong_AsLongLong( value );
			if (PyErr_Occurred() != NULL)
				goto _exit;
		} else if (a->descr->type_num == tUInt64) {
			v = (UInt64) PyLong_AsUnsignedLongLong( value );
			if (PyErr_Occurred() != NULL)
				goto _exit;
		} else {
			if (a->descr->type_num == tUInt32) {
				v = PyLong_AsUnsignedLong(value);
				if (v == -1 && PyErr_Occurred())
					goto _exit;
			} else {
				v = PyLong_AsLongLong(value);
				if (v == -1 && PyErr_Occurred())
					goto _exit;
			}
		}
		if (NA_overflow(a, v) < 0)
			return -1;
		NA_set_Int64(a, offset, v);
	} else if (PyFloat_Check(value)) {
		Float64 v = PyFloat_AsDouble(value);
		if (NA_overflow(a, v) < 0)
			return -1;
		NA_set_Float64(a, offset, v);
	} else if (PyComplex_Check(value)) {
		Complex64 vc;
		vc.r = PyComplex_RealAsDouble(value);
		vc.i = PyComplex_ImagAsDouble(value);
		if (NA_overflow(a, vc.r) < 0)
			return -1;
		if (NA_overflow(a, vc.i) < 0)
			return -1;
		NA_set_Complex64(a, offset, vc);
	} else if (PyObject_HasAttrString(value, "__tonumtype__")) {
		PyObject *type = NA_typeNoToTypeObject(a->descr->type_num);
		if (!type) goto _exit;
		value = PyObject_CallMethod(
			value, "__tonumtype__", "(N)", type);
		if (!value) goto _exit;
		rval = _setFromPythonScalarCore(a, offset, value, entries+1);
		Py_DECREF(value);
		goto _exit;
	} else if (PyString_Check(value)) {
		long size = PyString_Size(value);
		if ((size <= 0) || (size > 1)) {
			PyErr_Format( PyExc_ValueError, 
				      "NA_setFromPythonScalar: len(string) must be 1.");
			goto _exit;
		}
		NA_set_Int64(a, offset, *PyString_AsString(value));
	} else {
		PyErr_Format(PyExc_TypeError, 
			     "NA_setFromPythonScalar: bad value type.");
		goto _exit;
	}
	rval = 0;
  _exit:
	return rval;
}

static int
NA_setFromPythonScalar(PyArrayObject *a, long offset, PyObject *value)
{
	if (_checkOffset(a, offset) < 0)
		return -1;
	if (a->flags & WRITABLE)
		return _setFromPythonScalarCore(a, offset, value, 0);
	else {
		PyErr_Format(
			PyExc_ValueError, "NA_setFromPythonScalar: assigment to readonly array buffer");
		return -1;
	}
}

static int
NA_isPythonScalar(PyObject *o)
{
	int rval;
	rval =  PyInt_Check(o) || 
		PyLong_Check(o) || 
		PyFloat_Check(o) || 
		PyComplex_Check(o) ||
		(PyString_Check(o) && (PyString_Size(o) == 1));
	return rval;
}

unsigned long NA_elements(PyArrayObject  *a)
{
	int i;
	unsigned long n = 1;
	for(i = 0; i<a->nd; i++)
		n *= a->dimensions[i];
	return n;
}

/*
 * Local Variables:
 * mode: C
 * c-file-style: "python"
 * End:
 */
