/* Trim message catalogs. 
   Copyright (C) 1999, Enrique Zanardi <ezanard@debian.org>
   Based on code from msgfmt.c from the gettext package. msgfmt.c is:
   Copyright (C) 1995, 1996, 1997, 1998 Free Software Foundation, Inc.
   Written by Ulrich Drepper <drepper@gnu.ai.mit.edu>, April 1995.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gettext.h"
#include "domain.h"

extern struct loaded_domain *load_domain (char *domain_file);

static void write_table (struct loaded_domain *domain, int outputOrig);
static int write_table_2 (struct loaded_domain *domain,
		struct loaded_domain *ref_domain);

int main(int argc, char **argv) {
	struct loaded_domain *domain, *ref_domain;
	int outputOrig;

	outputOrig = 0;
	if (argc != 2 && argc != 3) {
		fprintf(stderr,"Usage: %s MO-file [Reference.mo]\n",argv[0]);
		exit(-1);
	}
	domain=load_domain(argv[1]);
	if (domain == NULL) {
		fprintf(stderr,"Error loading file: %s\n",argv[1]);
		exit(-1);
	}
	if (argc == 2) {
		write_table(domain, outputOrig);
		exit(0);
	} else { /* (argc == 3) */
		ref_domain=load_domain(argv[2]);
		if (ref_domain == NULL) {
			fprintf(stderr,"Error loading file: %s\n",argv[2]);
			exit(-1);
		}
		exit(write_table_2(domain, ref_domain));
	}
}

static void
write_table (struct loaded_domain *domain, int outputOrig)
{
  size_t cnt;
  nls_uint32 nstrings, offset;
  struct string_desc *descOrig, *descTrans;

  nstrings = domain->nstrings;

  /* Write the header out.  */
  fwrite (&nstrings, sizeof (nstrings), 1, stdout);

  /* Set offset to first byte after all the tables.  */
  offset = sizeof (nstrings) + nstrings * sizeof (offset);

  /* Write out length and starting offset for all original strings.  */
  descOrig=domain->orig_tab;
  descTrans=domain->trans_tab;
  for (cnt = 0; cnt < nstrings; ++cnt, ++descOrig, ++descTrans)
    {
      size_t len;
      const char *ptr;
      if (outputOrig)
        ptr = domain->data + descOrig->offset;
      else
        ptr = domain->data + descTrans->offset;
      len = strlen (ptr);

      fwrite (&offset, sizeof (offset), 1, stdout);
      offset += len + 1;
    }

  /* Now write the original strings.  */
  descOrig=domain->orig_tab;
  descTrans=domain->trans_tab;
  for (cnt = 0; cnt < nstrings; ++cnt, ++descOrig, ++descTrans)
    {
      size_t len;
      const char *ptr;
      if (outputOrig)
        ptr = domain->data + descOrig->offset;
      else
        ptr = domain->data + descTrans->offset;
      len = strlen (ptr);

      fwrite (ptr, len + 1, 1, stdout);
    }
}

static int
write_table_2 (struct loaded_domain *domain,
	       struct loaded_domain *ref_domain)
{
  size_t cnt;
  nls_uint32 nstrings_ref, nstrings, offset;
  char **messages;

  nstrings_ref = ref_domain->nstrings;
  nstrings = domain->nstrings;

  /* Write the header out.  */
  fwrite (&nstrings_ref, sizeof (nstrings), 1, stdout);
	       
  /* allocate the space for messages */
  if((messages = (char **)calloc (nstrings_ref, sizeof (char *))) == NULL) {
    return (-1); /* can not allocate memory */
  }

  /* Now Get the Reference strings.  */
  for (cnt = 0; cnt < nstrings_ref; ++cnt)
    {
      messages[cnt] = (char *) (ref_domain->data +
		      ref_domain->orig_tab[cnt].offset);
    }
  
  /* Now, array messages[] has all the Reference strings,
   * We use it to find the translated messages using 
   * domain->orig_tab and domain->trans_tab based on
   * binary search in the sorted array of messages.
   */
  
  /* These codes are based on dcgettext.c in GNU gettext */
  for (cnt = 0; cnt < nstrings_ref; ++cnt)
    {
      size_t bottom, top, act;
      const char *ref_ptr;

      ref_ptr = messages[cnt];
      bottom = 0;
      top = nstrings;

      while (bottom < top)
	{
	  int cmp_val;
  
	  act = (bottom + top) / 2;
	  cmp_val = strcmp (ref_ptr, (char *)(domain->data+
				     domain->orig_tab[act].offset));
	  if (cmp_val < 0)
	    top = act;
	  else if (cmp_val > 0)
	    bottom = act + 1;
	  else
	    break;
	}

      /* If an translation is found, replace messages[cnt] using it */
      if (bottom < top) {
	 messages[cnt] = (char *) domain->data +
				  domain->trans_tab[act].offset;
      }
    }
      
  /* Set offset to first byte after all the tables.  */
  offset = sizeof (nstrings) + nstrings_ref * sizeof (offset);
  
  /* Write out starting offset for all strings.  */
  for (cnt = 0; cnt < nstrings_ref; ++cnt)
    {
      size_t len;
	    
      len = strlen (messages[cnt]);
  
      fwrite (&offset, sizeof (offset), 1, stdout);
      offset += len + 1;
    }

  /* Now write all the strings.  */
  for (cnt = 0; cnt < nstrings_ref; ++cnt)
    {
      size_t len;

      len = strlen (messages[cnt]);
      
      fwrite (messages[cnt], len + 1, 1, stdout);
    }
  
  return (0); /* Success! */
}
