// soundeditor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <X11/keysym.h>
#include "application.h"
#include "combfilter.h"
#include "controller.h"
#include "converter.h"
#include "cmdstate.h"
#include "ellipfilt.h"
#include "datamodifier.h"
#include "fftfun.h"
#include "filecommand.h"
#include "filename.h"
#include "iallpole.h"
#include "interpolater.h"
#include "lowpassfilter.h"
#include "lpcdata.h"
#include "firfilter.h"
#include "formantfilter.h"
#include "formantsynth.h"
#include "optionsetter.h"
#include "pitchtracker.h"
#include "pvanalyzer.h"
#include "pvsynthesizer.h"
#include "reson.h"
#include "silencedetecter.h"
#include "sound.h"
#include "soundeditor.h"

Data *
SoundEditor::newModel() { return new Sound(); }

boolean
SoundEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_asciitilde:
		removeDC_Component();
		break;
	case XK_q:
		resonFilter();
		break;
	case XK_K:
		combFilter();
		break;
	case XK_H:
		ellipticalFilter();
		break;
	case XK_F:
		firFilter();
		break;
	case XK_brokenbar:	// unused symbol
		lpcFilter();
		break;
	case XK_KP_1:
		resynthesizeFromLPC();
		break;
	case XK_KP_2:
		resynthesizeFromPvoc();
		break;
	case XK_p:
		play();
		break;
	case XK_colon:
		record();
		break;
	case XK_BackSpace:
	case XK_Delete:
		stop();
		break;
	case XK_parenright:
		configureConverter();
		break;
	case XK_dollar:
		rescanForPeak();
		break;
	case XK_ampersand:
		changeFormat();
		break;
	case XK_k:
		changeSampleRate();
		break;
	case XK_4:
		FFTAnalysis();
		break;
	case XK_5:
		LPCAnalysis();
		break;
	case XK_6:
		pitchTrackAnalysis();
		break;
	case XK_7:
		LPCAndPitchTrackAnalysis();
		break;
	case XK_8:
		PVocAnalysis();
		break;
	case XK_9:
		findSilence();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

Status
SoundEditor::newFile() {
	return (SoundEditor::new_File(_controller) != nil);
}

// static constructor

Controller *
SoundEditor::new_File(Controller* controller) {
	SoundCreator sc;
	applyModifierUsing(controller->editor(), sc, controller, false);
	return sc.getNewController();
}

// protected utility used by base class save(), etc.

Modifier *
SoundEditor::getFileSaver(const char *dir, Data *d, const char *filename) {
	return (filename != NULL) ?
    	new SoundFileSaver(filename, dir, d) : new SoundFileSaver(dir, d);
}

// new or redefined public edit command methods for sounds only

Status
SoundEditor::removeDC_Component() {
	// run filter with cutoff starting at 20 hz, 60db down at 5hz
	EllipticalFilter e(currentSelection(), 20.0, 5.0);
	return applyModifier(e);
}

Status
SoundEditor::interpolate() {
	setCopyBuffer(currentSelection()->newData(1));
	Interpolater t(currentSelection(), copyBuffer(), true);
	return applyModifierToNew(t);
}

Status
SoundEditor::insertSpace() {
	SpaceInsert t(currentSelection(), /* asTime = */ true);
	return applyModifier(t);
}

Status
SoundEditor::changeLength() {
	LengthChanger d(model(), /* asDur = */ true);
	Status status = applyModifier(d);
	if(status)
		reselect();
	return status;
}

Status
SoundEditor::lowPassFilter() {
	LowPassFilter l(currentSelection());
	return applyModifier(l);
}

Status
SoundEditor::resonFilter() {
	Reson3 r(currentSelection());
	return applyModifier(r);
}

Status
SoundEditor::combFilter() {
	// output will be new floating point resized by comb object
	setCopyBuffer(
		new Sound(1, soundModel()->sRate(),
			currentSelection()->channels(), FloatData)
	);
	CombFilter c(currentSelection(), copyBuffer());
	return applyModifierToNew(c);
}

Status
SoundEditor::ellipticalFilter() {
	EllipticalFilter e(currentSelection());
	return applyModifier(e);
}

Status
SoundEditor::firFilter() {
	Data* coeffs = currentSource();
	Status status = Fail;
	if(coeffs != nil) {
		FIRFilter f(currentSelection(), currentSelection(), coeffs);
		status = applyModifier(f);
	}
	freeSource(coeffs);
	return status;
}

Status
SoundEditor::lpcFilter() {
	Data* lpc = currentSource();
	Status status = Fail;
	if(lpc != nil) {
		FormantFilter f(currentSelection(), currentSelection(), lpc);
		status = applyModifier(f);
	}
	freeSource(lpc);
	return status;
}

Status
SoundEditor::resynthesizeFromLPC() {
	Data* lpc = currentSource();
	Status status = Fail;
	if(lpc != nil) {
		FormantSynthesizer f(currentSelection(), lpc);
		status = applyModifier(f);
	}
	freeSource(lpc);
	return status;
}

Status
SoundEditor::play() {
	stop();
	Range oldChannels = _channels;	// save old value
	Range selected = currentRegion();
	bool wasSelected = selectionMade();
	setChannels(model()->channelRange());	// force all channels
	Converter* cvtr = Converter::getInstance();
	class ProgressAction *paction = _controller->getConverterProgressAction();
	Status st = ((Sound *) currentSelection())->play(cvtr, paction);
	// This puts us back at the beginning of the play region, 
	// regardless of whether we have a selection or not.
	_controller->showInsertPoint(selected.intMin(), oldChannels, true);
	if (wasSelected)
		_controller->showEditRegion(selected, oldChannels, false);	// do not expand view to display selection
	return st;
}

Status
SoundEditor::record() {
	Range oldChannels = _channels;	// save old value
	bool wasSelected = selectionMade();
	Range selected = currentRegion();
	setChannels(model()->channelRange());	// force all channels
	Converter* cvtr = Converter::getInstance();
	class ProgressAction *paction = _controller->getConverterProgressAction();
	Status st =  ((Sound *) currentSelection())->record(cvtr, paction);
	if (wasSelected)
		_controller->showEditRegion(selected, oldChannels);
	else
		_controller->showInsertPoint(selected.intMin(), oldChannels, true);
	return st;
}

Status
SoundEditor::stop() {
	return Converter::getInstance()->stop();
}

Status
SoundEditor::configureConverter() {
	Converter::getInstance()->configureHardware(_controller);
	return Succeed;
}

Status
SoundEditor::rescanForPeak() {
	boolean old = Data::deferRescan(false);
	model()->Notify();	// this automatically does a scan
	Data::deferRescan(old);
	return Succeed;
}

Status
SoundEditor::rescale() {
	Application::inform("Rescaling...");
	soundModel()->rescale();
	reselect();
	commandState()->Set(File_Is_Modified, true); // Until handled via modifier
	return Succeed;
}

Status
SoundEditor::changeFormat() {
	FormatChanger f(soundModel());
	Status status;
	if((status = applyModifier(f))) {
		reselect();
		_controller->setFileName(
			FileName::changeSuffix(
				_controller->fileName(), soundModel()->fileSuffix()
			)
		);
	}
	return status;
}

Status
SoundEditor::LPCAnalysis() {
	Status status = Succeed;
	InverseAllPole analyzer(currentSelection());
	if((status = applyModifier(analyzer))) {
		Controller *newctlr = new Controller(analyzer.getAnalysis());
		newctlr->display(_controller->world());
	}
	return status;
}

Status
SoundEditor::pitchTrackAnalysis() {
	Status status = Succeed;
	PitchTracker analyzer(currentSelection());
	if((status = applyModifier(analyzer))) {
		Controller *newctlr = new Controller(analyzer.getAnalysis());
		newctlr->display(_controller->world());
	}
	return status;
}

Status
SoundEditor::LPCAndPitchTrackAnalysis() {
	return LPCAnalysis() && pitchTrackAnalysis();
}

Status
SoundEditor::FFTAnalysis() {
	FFT_Function analyzer(currentSelection());
	Status status;
	if((status = applyModifier(analyzer))) {
		Controller *newctlr = new Controller(analyzer.getAnalysis());
		newctlr->display(_controller->world());
	}
	return status;
}

Status
SoundEditor::PVocAnalysis() {
	PVAnalyzer pva(currentSelection());
	Status status;
	if((status = applyModifier(pva))) {
		Controller *newctlr = new Controller(pva.getAnalysis());
		newctlr->display(_controller->world());
	}
	return status;
}

Status
SoundEditor::findSilence() {
	SilenceDetecter detecter(currentSelection());
	Status status;
	if ((status = applyModifier(detecter))) {
		_controller->showInsertPoint(
				currentInsert() + detecter.getSilenceOffset(),
				currentChannels());
	}
	else {
		Application::alert("None found.");
	}
	return status;
}

Status
SoundEditor::resynthesizeFromPvoc() {
	Data* pvdata = currentSource();
	Status status = Fail;
	if(pvdata != nil) {
		if(pvdata->isA(Pvoc_Data)) {
			PVSynthesizer pvs(currentSelection(), (PvocData *) pvdata);
			status = applyModifier(pvs);
		}
		else
			Application::alert("Selected source for pvoc resynthesis",
				"is not a Phase Vocoder data file.");
	}
	freeSource(pvdata);
	return status;
}

Status
SoundEditor::setDataOptions() {
	SoundOptionSetter options;
	return applyModifier(options);
}

Status
SoundEditor::setRawFileOptions() {
	RawSoundfileOptionSetter options;
	return applyModifier(options);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
SoundEditor::new_DataEditor1(Controller *c) {
	return new SoundEditor(c);
}

DataEditor *
SoundEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new SoundEditor(c, d);
}
