// filecommand.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "controller.h"
#include "data.h"
#include "editor.h"
#include "datafile.h"
#include "envelope.h"
#include "envelopeheader.h"
#include "filecommand.h"
#include "filename.h"
#include "lpcdata.h"
#include "lpcheader.h"
#include "pvocdata.h"
#include "query.h"
#include "request.h"
#include "formatrequester.h"
#include "sndconfig.h"
#include "sound.h"
#include "soundheader.h"

FileCommand::FileCommand(const char* filename)
	: theFileName(filename), theController(nil) {}

Status
FileCommand::configure(Controller *c) {
	theController = c;		// cache a ref to this for use
	return Super::configure(c);
}

int
FileCommand::apply() {
	int status = false;
	if(ok()) {
		Application::inform(message());
		status = doApply();
	}
	return status;
}

//********

class FileCreateRequester : public TitledRequester {
	friend class FileCreator;
	friend class EnvelopeFileCreator;
protected:
	FileCreateRequester(const char* title, FileCreator* fc)
		: TitledRequester(title), client(fc) {}
	redefined void configureRequest(Request *);
protected:
	FileCreator* client;
};

void
FileCreateRequester::configureRequest(Request* request) {
	String& name = client->fileName();
	request->appendValue("File name:", &name);
	request->appendValue("Length (in frames):", &client->newLength,
	                     PositiveIntegers);
}

//********

FileCreator::FileCreator(const char* filename, int length)
	: FileCommand(filename),
	  newController(nil), theData(nil), newLength(length) {}

// this could do length() != newLength

boolean
FileCreator::setData(Data *d) {
	theData = d;
	return (theData->length() != 0);
}

Data *
FileCreator::newData() {
	return theData;
}

int
FileCreator::doApply() {
	newController = new Controller(newData(), fileName());
	newController->display(controller()->world());
	return true;
}

//********

class SoundCreateRequester : public FormatRequester {
	friend class SoundCreator;
protected:
	SoundCreateRequester(SoundCreator *);
	redefined void configureRequest(Request *);
private:
	SoundCreator* client;
};

SoundCreateRequester::SoundCreateRequester(SoundCreator* sc)
	: FormatRequester(nil, "Sample format:", sc->format),
	  TitledRequester("Create New Soundfile:"),
	  client(sc) {}

void
SoundCreateRequester::configureRequest(Request* request) {
	String& name = client->fileName();
	request->appendValue("File name:", &name);
	request->appendValue("Duration (MM:SS):", &client->duration);
	request->appendValue("Sample rate:", &client->sampleRate, PositiveIntegers);
	request->appendValue("Channels:", &client->channels, PositiveIntegers);
	FormatRequester::configureRequest(request);
}

//********

SoundCreator::SoundCreator()
	: FileCreator(FileName::untitledName(Sound::defaultFileSuffix()), 0),
	  duration(1.0),
	  sampleRate(Sound::defaultSampleRate()),
	  channels(1),
	  format(Sound::defaultDataType()) {
}

Requester *
SoundCreator::createRequester() {
	return new SoundCreateRequester(this);
}

void
SoundCreator::initialize() {
	Super::initialize();
	duration = max(0.001, duration); // duration never zero
	okIf(setData(new Sound(duration, sampleRate, channels, DataType(format))));
}

//********

class LPCFileCreateRequester : public FileCreateRequester {
	friend class LPCFileCreator;
protected:
	LPCFileCreateRequester(LPCFileCreator *);
	redefined void configureRequest(Request *);
private:
	LPCFileCreator* creator() { return (LPCFileCreator *) client; }
};

LPCFileCreateRequester::LPCFileCreateRequester(LPCFileCreator* lfc)
	: FileCreateRequester("Create New LPC Datafile:", lfc) {}

void
LPCFileCreateRequester::configureRequest(Request* request) {
	FileCreateRequester::configureRequest(request);
	request->appendValue("Number of filter poles:", &creator()->nPoles,
			    Range(12, 64));
	request->appendValue("Frame rate (frames per second):",
	                     &creator()->frameRate, PositiveNumbers);
	request->appendValue("Source sample rate:", &creator()->sampleRate,
	                     PositiveIntegers);
}

//********

LPCFileCreator::LPCFileCreator()
	: FileCreator(FileName::untitledName(".lpc"), 32),
	  nPoles(LPCData::defaultNumberOfPoles()),
	  frameRate(LPCData::defaultFrameRate()),
	  sampleRate(Sound::defaultSampleRate()) {}

Requester *
LPCFileCreator::createRequester() {
	return new LPCFileCreateRequester(this);
}

void
LPCFileCreator::initialize() {
	Super::initialize();
	okIf(setData(new LPCData(dataLength(), nPoles, sampleRate, frameRate)));
}

//********

class PvocFileCreateRequester : public FileCreateRequester {
	friend class PvocFileCreator;
protected:
	PvocFileCreateRequester(PvocFileCreator *);
	redefined void configureRequest(Request *);
private:
	PvocFileCreator* creator() { return (PvocFileCreator *) client; }
};

PvocFileCreateRequester::PvocFileCreateRequester(PvocFileCreator* pfc)
	: FileCreateRequester("Create New PVoc Datafile:", pfc) {}

void
PvocFileCreateRequester::configureRequest(Request* request) {
	FileCreateRequester::configureRequest(request);
	request->appendValue("Frame rate (frames per second):",
	                     &creator()->frameRate, PositiveNumbers);
	request->appendValue("Source sample rate:", &creator()->sampleRate,
	                     PositiveIntegers);
	request->appendChoice("Number of frequency bands (N/2):",
	                      "|64|128|256|512|1024|2048|4096|8192|16384|",
	                      &creator()->nBands,
	                      true);
}

//********

PvocFileCreator::PvocFileCreator()
	: FileCreator(FileName::untitledName(".pv"), 32),
	  frameRate(PvocData::defaultFrameRate()),
	  sampleRate(Sound::defaultSampleRate()),
	  nBands(PvocData::defaultNumberOfBands() >> 6) {}

Requester *
PvocFileCreator::createRequester() {
	return new PvocFileCreateRequester(this);
}

void
PvocFileCreator::initialize() {
	Super::initialize();
	int npoints = nBands << 6; // maps {1, 2, 4, ..} into {64, 128, ...}
	okIf(setData(new PvocData(dataLength(),
				  (npoints + 1) * 2,
				  sampleRate,
				  frameRate)));
}

//********

EnvelopeFileCreator::EnvelopeFileCreator()
	: FileCreator(FileName::untitledName(".evp"), 512) {}

Requester *
EnvelopeFileCreator::createRequester() {
	return new FileCreateRequester("Create New Envelope:", this);
}

void
EnvelopeFileCreator::initialize() {
	Super::initialize();
	okIf(setData(new Envelope(dataLength())));
}

//********

class FileAccessRequester : public TitledRequester {
	friend class FileAccessCommand;
protected:
	FileAccessRequester(const char* title, FileAccessCommand* fac)
		: TitledRequester(title), client(fac) {}
	redefined Request* createRequest();
protected:
	FileAccessCommand* client;
};

Request *
FileAccessRequester::createRequest() {
	return new FileRequest(
		requestTitle(), &client->defaultDir,
		client->defaultSuffixes(), createDelegate()
	);
}

//********

FileAccessCommand::FileAccessCommand(const char* dir)
	: FileCommand(""), defaultDir(dir) {}

//********

class FileSaveRequester : public FileAccessRequester {
	friend class FileSaver;
protected:
	FileSaveRequester(FileSaver* fs)
		: FileAccessRequester("Save to File:", fs) {}
	FileSaveRequester(FileSaver* fs, const char *title)
		: FileAccessRequester(title, fs) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	FileSaver* saver() { return (FileSaver *) client; }
};

void
FileSaveRequester::configureRequest(Request* request) {
	request->appendChoice("File format:", "|Raw (no header)|Use header|",
		&saver()->headerFormat, true);
}

boolean
FileSaveRequester::confirmValues() {
	FileSaver* fs = saver();
	// directory name has file appended upon return from browser
	fs->setFileName(fs->defaultDirectory());
	fs->isNewFile = (DiskFile::exists(fs->fileName()) == false);
	boolean confirmed = true;
	if(!fs->isNewFile) {
		Response r = Application::choice(
			scat("The file ", fs->fileName(), " already exists."),
			"Do you wish to overwrite it?", nil, Cancel,
			"force new header", "keep old header", "cancel");
		switch (r) {
		case Yes:
			fs->forceHeader = true; break;
		case No:
			fs->forceHeader = false; break;
		case Cancel:
		default:
			fs->forceHeader = false;
			confirmed = false;
			break;
		};
	}
	return confirmed;
}

//********

extern const char *scat(const char *, const char *);
extern const char *scat(const char *, const char *, const char *);

FileSaver::FileSaver(const char* name, const char* defaultDir, Data* data)
		: FileAccessCommand(defaultDir),
		  dataToSave(data), forceHeader(false) {
	// dont allow saving to "untitled_*" or "tmp_*" temp names
	if(!FileName::isTempName(name)) {
		// prepend default directory to name if none
		const char* filename = (FileName::isFullPathName(name)) ?
			name : scat(defaultDir, "/", name);
		isNewFile = (DiskFile::exists(filename) == false);
		setFileName(filename);
		initialize();
	}
	else isNewFile = true;
}

FileSaver::FileSaver(const char* dir, Data* data) : FileAccessCommand(dir),
	dataToSave(data), headerFormat(WithHeader),
	isNewFile(true), forceHeader(false) {}

Requester *
FileSaver::createRequester() {
	return new FileSaveRequester(this);
}

void
FileSaver::initialize() {
	Super::initialize();
}

const char *
FileSaver::message() {
	return isNewFile ?
		"Writing new file to disk..." : "Overwriting file on disk...";
}

int
FileSaver::doApply() {
	DataFile file;
	// File truncated if new header type is to be forced
	file.open(fileName(), (isNewFile || forceHeader) ? "w" : "r+");
	int status = (file.writable() && dataToSave->write(&file));
	if(status)
		controller()->setFileName(fileName());	// rename file to new name
	else if(isNewFile)
		file.remove();		// if new file was unwritable, delete it
	return status;
}

//********

class SoundFileSaveRequester : public FileSaveRequester {
	friend class SoundFileSaver;
protected:
	SoundFileSaveRequester(SoundFileSaver* s) : FileSaveRequester(s) {}
	redefined void configureRequest(Request *);
private:
	SoundFileSaver* saver() { return (SoundFileSaver *) client; }
};

void
SoundFileSaveRequester::configureRequest(Request* request) {
	request->appendChoice("Sound file header type:",
		"|Raw (No Header)|Snd/au|Hybrid|BSD/IRCAM|AIFF-C|WAVE|",
		&saver()->soundHeaderFormat,
		true
	);
}
	
//********

SoundFileSaver::SoundFileSaver(const char* dir, Data* data)
	: FileSaver(dir, data),
	  soundHeaderFormat(SoundHeader::defaultHeaderType()) {}

SoundFileSaver::SoundFileSaver(const char* name, const char* dir, Data* data)
	: FileSaver(name, dir, data),
	  soundHeaderFormat(SoundHeader::defaultHeaderType()) {}

Requester *
SoundFileSaver::createRequester() {
	return new SoundFileSaveRequester(this);
}

void
SoundFileSaver::initialize() {
	Super::initialize();
	SoundHeader::setDefaultHeaderType(SoundHeader::Type(soundHeaderFormat));
}

//********

LPCFileSaver::LPCFileSaver(const char* dir, Data* data)
	: FileSaver(dir, data) {}

LPCFileSaver::LPCFileSaver(const char* name, const char* dir, Data* data)
	: FileSaver(name, dir, data) {}

void
LPCFileSaver::initialize() {
	Super::initialize();
	LPCHeader::setDefaultHeaderType(
		(headerFormat == WithHeader) ? LPCHeader::With : LPCHeader::None
	);
}

//********

EnvelopeFileSaver::EnvelopeFileSaver(const char* dir, Data* data)
	: FileSaver(dir, data) {}

EnvelopeFileSaver::EnvelopeFileSaver(const char* name, const char* dir, Data* data)
	: FileSaver(name, dir, data) {}

void
EnvelopeFileSaver::initialize() {
	Super::initialize();
	EnvelopeHeader::setDefaultHeaderType(
		(headerFormat == WithHeader) ?
		    EnvelopeHeader::With : EnvelopeHeader::None
	);
}

//********

class DataDumpRequester : public FileAccessRequester {
	friend class DataDumper;
protected:
	DataDumpRequester(DataDumper* d)
		: FileAccessRequester(
			"Dump selection's frame/sample data as ASCII file:", d) {}
	redefined void configureRequest(Request *) {}
	redefined boolean confirmValues();
private:
	DataDumper* saver() { return (DataDumper *) client; }
};

boolean
DataDumpRequester::confirmValues() {
	// directory name has file appended upon return from browser
	saver()->setFileName(saver()->defaultDirectory());
	return true;
}

//********

Requester *
DataDumper::createRequester() {
	return new DataDumpRequester(this);
}


const char *
DataDumper::message() {
	return "Dumping data to file...";
}

int
DataDumper::doApply() {
	int status = true;
	DataFile file;
	file.open(fileName(), "w+");
	if ((status = file.writable()) == true)
		dataToSave->print(file);
	return status;
}

//********

class FileOpenRequester : public FileAccessRequester {
	friend class FileOpener;
protected:
	FileOpenRequester(FileOpener* f) : FileAccessRequester(nil, f) {}
	redefined const char* requestTitle();
	redefined void configureRequest(Request*);
	redefined boolean confirmValues();
private:
	FileOpener* opener() { return (FileOpener *) client; }
};

// this overrides TitledRequester::requestTitle()

const char*
FileOpenRequester::requestTitle() { return opener()->theTitle; }

void
FileOpenRequester::configureRequest(Request* request) {
	request->appendValue("Time to skip (MM:SS): ", &opener()->skipTime);
	request->appendValue("Duration to read (MM:SS, 0 = all):",
	                     &opener()->duration);
}

boolean
FileOpenRequester::confirmValues() {
	// directory name has file appended upon return from browser
	client->setFileName(client->defaultDirectory());
	return true;
}

//********

FileOpener::FileOpener(const char* dir, const char* title)
	: FileAccessCommand(dir), theTitle(title),
	  skipTime(0), duration(0), newController(nil) {}

Requester *
FileOpener::createRequester() {
	return new FileOpenRequester(this);
}

const char *
FileOpener::message() {
	return "Opening file...";
}

int
FileOpener::doApply() {
	DataFile* file = new DataFile(fileName(), "r", skipTime, duration);
	file->ref();
	if(file->is_open()) {
		if(file->readable())
			newController = Controller::create(file);
		else
			Application::error(scat("Unable to read file ", file->name()));
	}
	if(newController)
		newController->display(controller()->world());
	Resource::unref(file);
	return (newController != nil);
}
