/* 
   MultiSync SyncML plugin - Implementation of SyncML 1.1 and 1.0
   Copyright (C) 2002-2003 Bo Lincoln <lincoln@lysator.liu.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation;

   In addition, as a special exception, Bo Lincoln <lincoln@lysator.liu.se>
   gives permission to link the code of this program with
   the OpenSSL library (or with modified versions of OpenSSL that use the
   same license as OpenSSL), and distribute linked combinations including
   the two.  You must obey the GNU General Public License in all
   respects for all of the code used other than OpenSSL.  If you modify
   this file, you may extend this exception to your version of the
   file, but you are not obligated to do so.  If you do not wish to
   do so, delete this exception statement from your version.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
   IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE FOR ANY
   CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES 
   WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
   ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF 
   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

   ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS, 
   COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS 
   SOFTWARE IS DISCLAIMED.
*/

/*
 *  $Id: syncml_ssl.c,v 1.5 2004/02/09 18:53:31 lincoln Exp $
 */

#include "config.h"
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <openssl/ssl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <openssl/dh.h>
#include "syncml_engine.h"
#include "syncml_plugin.h"

extern gboolean multisync_debug;

DH *syncml_strong_dh2048() {
  static unsigned char dh2048_p[]={
    0xF6,0x42,0x57,0xB7,0x08,0x7F,0x08,0x17,0x72,0xA2,0xBA,0xD6,
    0xA9,0x42,0xF3,0x05,0xE8,0xF9,0x53,0x11,0x39,0x4F,0xB6,0xF1,
    0x6E,0xB9,0x4B,0x38,0x20,0xDA,0x01,0xA7,0x56,0xA3,0x14,0xE9,
    0x8F,0x40,0x55,0xF3,0xD0,0x07,0xC6,0xCB,0x43,0xA9,0x94,0xAD,
    0xF7,0x4C,0x64,0x86,0x49,0xF8,0x0C,0x83,0xBD,0x65,0xE9,0x17,
    0xD4,0xA1,0xD3,0x50,0xF8,0xF5,0x59,0x5F,0xDC,0x76,0x52,0x4F,
    0x3D,0x3D,0x8D,0xDB,0xCE,0x99,0xE1,0x57,0x92,0x59,0xCD,0xFD,
    0xB8,0xAE,0x74,0x4F,0xC5,0xFC,0x76,0xBC,0x83,0xC5,0x47,0x30,
    0x61,0xCE,0x7C,0xC9,0x66,0xFF,0x15,0xF9,0xBB,0xFD,0x91,0x5E,
    0xC7,0x01,0xAA,0xD3,0x5B,0x9E,0x8D,0xA0,0xA5,0x72,0x3A,0xD4,
    0x1A,0xF0,0xBF,0x46,0x00,0x58,0x2B,0xE5,0xF4,0x88,0xFD,0x58,
    0x4E,0x49,0xDB,0xCD,0x20,0xB4,0x9D,0xE4,0x91,0x07,0x36,0x6B,
    0x33,0x6C,0x38,0x0D,0x45,0x1D,0x0F,0x7C,0x88,0xB3,0x1C,0x7C,
    0x5B,0x2D,0x8E,0xF6,0xF3,0xC9,0x23,0xC0,0x43,0xF0,0xA5,0x5B,
    0x18,0x8D,0x8E,0xBB,0x55,0x8C,0xB8,0x5D,0x38,0xD3,0x34,0xFD,
    0x7C,0x17,0x57,0x43,0xA3,0x1D,0x18,0x6C,0xDE,0x33,0x21,0x2C,
    0xB5,0x2A,0xFF,0x3C,0xE1,0xB1,0x29,0x40,0x18,0x11,0x8D,0x7C,
    0x84,0xA7,0x0A,0x72,0xD6,0x86,0xC4,0x03,0x19,0xC8,0x07,0x29,
    0x7A,0xCA,0x95,0x0C,0xD9,0x96,0x9F,0xAB,0xD0,0x0A,0x50,0x9B,
    0x02,0x46,0xD3,0x08,0x3D,0x66,0xA4,0x5D,0x41,0x9F,0x9C,0x7C,
    0xBD,0x89,0x4B,0x22,0x19,0x26,0xBA,0xAB,0xA2,0x5E,0xC3,0x55,
    0xE9,0x32,0x0B,0x3B,
  };
  static unsigned char dh2048_g[]={
    0x02,
  };
  DH *dh;
  
  if ((dh=DH_new()) == NULL) return(NULL);
  dh->p=BN_bin2bn(dh2048_p,sizeof(dh2048_p),NULL);
  dh->g=BN_bin2bn(dh2048_g,sizeof(dh2048_g),NULL);
  if ((dh->p == NULL) || (dh->g == NULL))
    { DH_free(dh); return(NULL); }
  return(dh);
}


// Returns -1 on timeout, 0 on error, read bytes otherwise
int syncml_ssl_read(syncml_state *state, char *data, int len, int timeout) {
  struct timeval tv;
  fd_set rset, wset, xset;
  int lenleft = len;
  int fd = state->connfd;
  gboolean read = TRUE;
  if (fd < 0)
    return(0);

  while (lenleft > 0) {
    int ret;
    ret = SSL_read(state->ssl, data+len-lenleft, lenleft); 
    if (ret > 0) {
      lenleft -= ret;
    } else {
      int err = SSL_get_error(state->ssl, ret);
      if (err == SSL_ERROR_WANT_READ)
	read = TRUE;
      else if (err == SSL_ERROR_WANT_WRITE)
	read = FALSE;
      else
	return(0);
    }
    FD_ZERO(&rset);
    FD_ZERO(&wset);
    FD_ZERO(&xset);
    if (read)
      FD_SET(fd, &rset);
    else
      FD_SET(fd, &wset);
    tv.tv_sec = timeout;
    tv.tv_usec = 0;
    if (lenleft > 0) {
      if (!select(fd+1,  &rset, &wset, &xset, &tv)) {
	return(-1); // Timeout
      }
    }
  }
  return(len);
}

// Returns -1 on timeout, 0 on error, written bytes otherwise
int syncml_ssl_write(syncml_state *state, char *data, int len, int timeout) {
  struct timeval tv;
  fd_set rset, wset, xset;
  int lenleft = len;
  int fd = state->connfd;
  gboolean write = TRUE;
  if (fd < 0)
    return(0);

  while (lenleft > 0) {
    int ret;
    ret = SSL_write(state->ssl, data+len-lenleft, lenleft); 
    if (ret > 0) {
      lenleft -= ret;
    } else {
      int err = SSL_get_error(state->ssl, ret);
      if (err == SSL_ERROR_WANT_WRITE)
	write = TRUE;
      else if (err == SSL_ERROR_WANT_READ)
	write = FALSE;
      else
	return(0);
    }
    FD_ZERO(&rset);
    FD_ZERO(&wset);
    FD_ZERO(&xset);
    if (write)
      FD_SET(fd, &wset);
    else
      FD_SET(fd, &rset);
    tv.tv_sec = timeout;
    tv.tv_usec = 0;
    if (lenleft > 0) {
      if (!select(fd+1,  &rset, &wset, &xset, &tv)) {
	return(-1); // Timeout
      }
    }
  }
  return(len);
}

// Generate a private RSA key and a self-signed certificate
void syncml_gen_rsa_keycert(char *keyfile, char *certfile) {
  int tofd[2];
  char certinfo[] = "--\nSome province\nSome city\nSome org\nSome section\nlocalhost.localdomain\nroot@localhost\n";
  char buf;

  pipe(tofd);
  if (!fork()) {
    dup2(tofd[0],0);
    execlp("openssl", "openssl", "req" , "-newkey", "rsa:1024", 
	   "-keyout", keyfile, 
	   "-nodes", "-x509", "-days", "365", "-out", certfile, NULL);
    exit(0);
  }
  write(tofd[1], certinfo, strlen(certinfo)+1);
  close(tofd[1]);
  wait(NULL);
  chmod(keyfile, 0600);
  chmod(certfile, 0600);
}

// Call to free an SSL connection
void syncml_ssl_disconnect(syncml_state *state){
  if (state->ssl) {
    SSL_shutdown(state->ssl);
    SSL_free(state->ssl);
    state->ssl = NULL;
  }
}

// Call to free any remaining SSL data
void syncml_ssl_exit(syncml_state *state) {
  syncml_ssl_disconnect(state);
  if (state->sslctx) {
    SSL_CTX_free(state->sslctx);
    state->sslctx = NULL;
  }
}


// Set up a SSL server on a connected fd.
gboolean syncml_ssl_server_connect(syncml_state *state) {
  SSL *ssl;
  if (state->connfd < 0)
    return(FALSE);
  ssl = SSL_new(state->sslctx);
  if (!ssl) {
    dd(printf("No SSL.\n"));
    return(FALSE);
  }
  if (!SSL_set_fd(ssl, state->connfd)) {
    dd(printf("No FD.\n"));
    return(FALSE);
  }
  SSL_set_accept_state(ssl);
  state->ssl = ssl;
  return(TRUE);
}


gboolean syncml_ssl_client_connect(syncml_state *state) {
  SSL *ssl;
  if (state->connfd < 0)
    return(FALSE);
  ssl = SSL_new(state->sslctx);
  if (!ssl) {
    dd(printf("No SSL.\n"));
    return(FALSE);
  }
  if (!SSL_set_fd(ssl, state->connfd)) {
    dd(printf("No FD.\n"));
    return(FALSE);
  }
  SSL_set_connect_state(ssl);
  state->ssl = ssl;
  return(TRUE);
}


gboolean syncml_ssl_init_server(syncml_state *state) {
  SSL_CTX *ctx;
  
  char *keypath = NULL;
  char *certpath = NULL;
  keypath = g_strdup_printf("%s/syncmlsslkey.pem", 
			    syncml_get_datapath(state->userdata));
  certpath = g_strdup_printf("%s/syncmlsslcert.pem", 
			     syncml_get_datapath(state->userdata));
  
  ctx = SSL_CTX_new(SSLv23_method());
  if (!ctx) {
    dd(printf("Could not create CTX.\n"));
    g_free(keypath);
    g_free(certpath);
    return(FALSE);
  }
  SSL_CTX_set_tmp_dh(ctx, syncml_strong_dh2048());
  SSL_CTX_set_options(ctx, SSL_OP_SINGLE_DH_USE);
  if (!SSL_CTX_use_PrivateKey_file(ctx, keypath, SSL_FILETYPE_PEM)) {
    syncml_gen_rsa_keycert(keypath, certpath);
    if (!SSL_CTX_use_PrivateKey_file(ctx, keypath, SSL_FILETYPE_PEM)) {
      g_free(keypath);
      g_free(certpath);
      dd(printf("Could not load private key."));
      return(FALSE);
    }
  }
  if (!SSL_CTX_use_certificate_file(ctx, certpath, SSL_FILETYPE_PEM)) {
    dd(printf("Could not load certificate.\n"));
    g_free(keypath);
    g_free(certpath);
    return(FALSE);
  }
  state->sslctx = ctx;
  g_free(keypath);
  g_free(certpath);
  return(TRUE);
}

gboolean syncml_ssl_init_client(syncml_state *state) {
  SSL_CTX *ctx;
  
  ctx = SSL_CTX_new(SSLv23_method());
  if (!ctx) {
    dd(printf("Could not create CTX.\n"));
    return(FALSE);
  }
  SSL_CTX_set_tmp_dh(ctx, syncml_strong_dh2048());
  SSL_CTX_set_options(ctx, SSL_OP_SINGLE_DH_USE);
  state->sslctx = ctx;
  return(TRUE);
}

// Initialization, called only once.
void syncml_ssl_init(void) {
  SSL_library_init();
  SSL_load_error_strings();
}
